# This file is part of ModPipe, Copyright 1997-2010 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.


from modeller import *

def calculate_pair_superposition(env, aln, chain1, chain2, typ):
    """Create a pairwise superposition between two structures given an
    alignment and report various numbers"""

    # Read in the models
    m0 = model(env, file=aln[0].atom_file, model_segment=aln[0].range)
    m1 = model(env, file=aln[1].atom_file, model_segment=aln[1].range)
    sel = selection(m0).only_atom_types('CA')

    # Compute sequence identity from the alignment
    eqvidpos = aln[0].get_num_equiv(aln[1])

    # Print a header
    columns = ['code1', 'len1', 'alnlen1', 'code2', 'len2',
               'alnlen2', 'eqvidpos', 'seqid', 'mod_grmsd', 'mod_geqvp',
               '(mod_cutoff, mod_cutoff_rmsd, mod_cutoff_eqvp)*']

    # Format and print results
    results = "%2s> %s %5d %5d %-5s %5d %5d %5d " % (typ, aln[0].code,
              len(chain1.residues), len(m0.residues),
              aln[1].code, len(chain2.residues), len(m1.residues),
              eqvidpos)

    # Now add the modeller numbers
    r = sel.superpose(m1, aln, rms_cutoff=3.5)
    results = results +  "%5.0f %8.4f %6d %5.2f %8.4f %6d " % \
              (100*eqvidpos/r.num_equiv_pos, r.rms, r.num_equiv_pos,
               3.5, r.cutoff_rms, r.num_equiv_cutoff_pos)

    cuts = [1.0, 2.0, 3.0, 4.0, 5.0, 8.0]
    for c in cuts:
        r = sel.superpose(m1, aln, rms_cutoff=c)
        results = results +  "%5.2f %8.4f %6d " % ( c, r.cutoff_rms,
                                                   r.num_equiv_cutoff_pos)

    header = '# ' + ' '.join(columns)
    return header, results
