#!/usr/bin/perl

use strict;
use Getopt::Long;
use File::Basename;
use Cwd;

# --- Load local modules
use PLLib::Utils;
use PLLib::Sequence;
use PLLib::Modeller;

use MPLib::Version;
use MPLib::Binaries;
use MPLib::MPInit;
use MPLib::MPUtils;
use MPLib::MPModules;

# --- Get command line options
$Getopt::Long::ignorecase = 0;

my ( $runname, $rundir, $nodes, $disks, $modpipeclusterbase);

my @final_modby = ();

my ( $hitsmode, $templatepdbid, $templatepdbchain,$enddir,$nodes_tb, $template_option );

my ($help,$tmpdir, $tmpdirroot,$pdb_file,$tsvmod_flag);

# --- Command-line options -- reflects subset used in ModWebd
GetOptions (
           "runname=s"                 =>      \$runname,
           "run_directory=s"           =>      \$rundir,
           "end_directory=s"           =>      \$enddir,
           "nodes=s"                   =>      \$nodes,
           "nodes_tb=s"                =>      \$nodes_tb,
           "tmpdir=s"                  =>      \$tmpdir,
           "tmpdirroot=s"              =>      \$tmpdirroot,
           "disks=s"                   =>      \$disks,
           "score_by_tsvmod=s"         =>      \$tsvmod_flag,
           "modpipeclusterbase=s"      =>      \$modpipeclusterbase,
           "final_models_by=s"         =>      \@final_modby,
           "hits_mode=s"               =>      \$hitsmode,
           "pdb_id=s"                  =>      \$templatepdbid,
           "pdb_chain=s"               =>      \$templatepdbchain,
           "template_option=s"	       =>      \$template_option,
           "pdb_file=s"	               =>      \$pdb_file,
           "help"                      =>      \$help,
           "version"                   => sub { VersionMessage() },
           );

   
# --- Check command line options
if ( $help ){
   &usage;
   exit 0;
}

# --- Get Program name
my $subrname = GetSubrName();


  # -- Change to rundir
  chdir($rundir);

  # -- Create the SGE job file
  # -- Open the SGE job file
  my $sgefile = "runTB.csh";
  my $sgefh = OpenNewFile($sgefile);
  if (!$tmpdir) {
     if (!$tmpdirroot) {
         $tmpdir="/scratch/ModWeb20d/${runname}/";
     } else {
         $tmpdir="$tmpdirroot/${runname}";
     }
  }

  # -- Write SGE job file
  my $final_modby_str = join ",", @final_modby;
  $final_modby_str =~ s/\s//g;
  my $return=WriteSGETB($sgefh, $runname, $rundir, $nodes, $nodes_tb, $disks, $modpipeclusterbase,
             $final_modby_str, $hitsmode, $templatepdbid, $templatepdbchain,$template_option,$tmpdir,$pdb_file,$tsvmod_flag);

  # -- Close the SGE job file handle
  close($sgefh);
  if ($return != 1) {
    die "${subrname}__E> Error writing $sgefile\n";
  }

  
# -- Submit job to the queue
  # -- Verify if qsub is in the path
  my $qsubexe = `which qsub 2> /dev/null` or
    die "${subrname}__E> Cannot find qsub in path\n";
  chomp $qsubexe;

  # -- Submit job and collect jobid
  warn "${subrname}__M> Submitting job to cluster\n";
  my $jobid = `qsub $sgefile`;
  chomp $jobid;
  $jobid = (split(" ",$jobid))[2];
  $jobid =~ s/\..*//;
  warn "${subrname}__M> Job successfully submitted: $jobid\n";

  # -- wait for job completion and collect results
  while ( 1 ){
    # -- Sleep for a while
    sleep 120; 

    # -- Check status of job
    my $qstatout = `qstat -j $jobid 2>&1`;
    last if ($qstatout =~ /^Following jobs do not exist/);
  }

  # -- Post process data
  warn "${subrname}__M> Job $jobid completed.\n";
#--- exit
exit 0;

# -- Subroutines

# --- Usage
sub usage {
print <<EOF;

${0}:

Interface to ModWebd for queue submission of StrucImpact.pl

Options:

      --pdb_id                  Four-character PDB code of the template of
                                interest (required).
      --pdb_chain               Single-character chain code.  If not specified,
                                the calculations will be performed for each non-
                                redundant chain.
      --pdb_file 		Full path of pdb file, needed in case the pdb file
                                is not in the standard pdb repository
      --runname                 A runname for the ModWeb run. If not
                                specified it will create one of the form:
                                ModWeb0b-(time). A directory of this name
                                will be created in the current working path.
      --run_directory           The directory from where the jobs should be
                                run. All data will be written into this
                                directory. This should be specified with 
                                absolute path.
                                Default: [PWD]/<runname>
      --tmpdirroot              TMPDIRRoot, default: /scratch/ModWeb20d/
                                will be expanded to TMPDIRROOT/<runname>
      --tmpdir                  TMPDIR, default: /scratch/ModWeb20d/<runname>
				overwrites tmpdirroot
      --final_models_by         Scheme to choose the final models by. Possible
                                values are LONGEST_DOPE, LONGEST_GA341, SEQID
                                GA341, DOPE, MPQS, TSVMod ALL.  Multiple options can
                                be specified by multiple copies of the command
                                line switch.  For example, "--final_models_by
                                LONGEST_DOPE --final_models_by SEQID" will
                                return two models.
                                Default: LONGEST_DOPE
      --modpipeclusterbase      The modpipe base accessible from the cluster.
				Will default to the base in this ModPipe installation.
      --disks                   The names of the disk-complexes defined in SGE.
                                Specify as a string enclosed within double quotes.
                                Will not proceed without this option.
      --nodes                   The names of the node-complexes defined in SGE.
                                Specify as a string enclosed within double quotes.
                                Will not proceed without this option.
MODPIPE Options:
  These are options that you would normally specify on the command-line of
  ModPipe.pl. These will form the basis for the SGE task included in the
  SGE job file.
      --exit_stage              Choose the exit stage for the program. You can
                                quit after one of the following: PROFILE,
                                ALIGNMENTS, MODELS.
                                Default: NONE
      --hits_mode               Mode for calculating template hits. It is a
                                three-letter code containing 1 (true) or 0
                                (false) for each of Seq-Seq, Prf-Seq and
                                Prf-Prf, respectively. For instance, to
                                calculate Seq-Seq and Prf-Prf, set it to 101.
                                Default: 111
      --evalue_hits             The E-value threshold to get hits against
                                template databases. This value controls hits
                                from all three searches.
                                Default: 1.0
      --score_by_tsvmod		[ON],OFF (set off when tsvmod is not installed)
      --clean_up                Flag to clean up the temporary directory
                                after all operations. Can be OFF or ON.
                                Default: ON

EOF
}

