#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2010 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from modeller import *
from optparse import OptionParser
import modpipe.version
import modpipe.pdbutils
import sys, os

def main():

    # Parse command line options
    parser = OptionParser(version=modpipe.version.message())

    # Set defaults
    parser.set_usage("""
 This script takes an alignment file in PIR format and trims the
 overhangs so that they are not modeled.

 Usage: %prog [options]

 Run `%prog -h` for help information
 """)

    parser.set_defaults(inpfile='',
                        editcode='',
                        outfile='overhang')

    # Populate options list
    parser.add_option("-i", "--input_file",
                 dest="inpfile",
                 type='string',
                 help="""File containing the alignment to be trimmed.
                      This is a mandatory option.""",
                 metavar="FILE")
    parser.add_option("-e", "--edit_align_code",
                 dest="editcode",
                 type='string',
                 help="""Align code of the sequence to be trimmed.
                      Mandatory option.""",
                 metavar="CODE")
    parser.add_option("-o", "--output_file",
                 dest="outfile",
                 help="""Name of the file to write out the trimmed alignment.""",
                 metavar="FILE")
    pdb = modpipe.pdbutils.get_pdb_repository()
    parser.add_option("-x", "--pdb_repository",
                      dest="pdbrep", type='string',
                      help="""Directories containing PDB files. The default
                              value is """ + pdb, default=pdb,
                      metavar="DIR")

    # Check mandatory options
    opts, args = parser.parse_args()

    if not opts.inpfile or not opts.editcode:
        parser.print_help()
        sys.exit(1)

    if not os.path.isfile(opts.inpfile):
        parser.error("""Cannot proceed without an input file""")

    # -- Initialize some modeller stuff
    log.verbose()
    env = environ()
    env.io.atom_files_directory = opts.pdbrep

    aln = alignment(env, file=opts.inpfile, align_codes='all',
                 alignment_format='PIR')
    aln.edit(edit_align_codes=opts.editcode, base_align_codes='rest',
          min_base_entries=1, overhang=3)
    aln.write(file=opts.outfile, alignment_format='PIR')

if __name__ == "__main__":
    main()
