#!/usr/bin/python
# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

from modeller import *
from optparse import OptionParser
from shutil import rmtree
from modpipe.tmalign import TMAlign
import modpipe.version
import sys, os, subprocess, tempfile, re

def main():
    """Get and parse the command line options."""

    # Parse command line options
    parser = OptionParser(version=modpipe.version.message())

    # Set defaults
    parser.set_usage("""
 This script takes two PDB protein chains and aligns them using TMalign. The
 resulting alignment is then also cleaned up to make it Modeller-friendly.
 The various structural overlap numbers are also reported.

 Usage: %prog [options]

 Run `%prog -h` for help information
 """)

    parser.set_defaults(pdb1='',
                        chn1='',
                        pdb2='',
                        chn2='',
                        tminpf='',
                        tmoutf='',
                        tmalif='',
                        )

    # Populate options list
    parser.add_option("-p", "--pdb1",
                 dest="pdb1",
                 type='string',
                 help="""PDB code of the first structure.
                      This is a mandatory option.""",
                 metavar="PDB")
    parser.add_option("-c", "--chn1",
                 dest="chn1",
                 type='string',
                 help="""PDB chain identifier of the first structure.
                      If not specified it will take the first chain.
                      """,
                 metavar="CHN")
    parser.add_option("-q", "--pdb2",
                 dest="pdb2",
                 type='string',
                 help="""PDB code of the second structure.
                      This is a mandatory option.""",
                 metavar="PDB")
    parser.add_option("-d", "--chn2",
                 dest="chn2",
                 type='string',
                 help="""PDB chain identifier of the second structure.
                      If not specified it will take the first chain.
                      """,
                 metavar="CHN")
    pdb = modpipe.pdbutils.get_pdb_repository(include_local=True)
    parser.add_option("-x", "--pdb_repository",
                      dest="pdbrep", type='string',
                      help="""Directories containing PDB files. The default
                              value is """ + pdb, default=pdb,
                      metavar="DIR")
    parser.add_option("-i", "--tm_input_file",
                      dest="tminpf", type='string',
                      help="""Input file that already contains a pre-calculated
                              TMalign alignment. If specified, this file will
                              be used instead of running TMalign.""",
                      metavar="FILE")
    parser.add_option("-o", "--tm_output_file",
                 dest="tmoutf",
                 type='string',
                 help="""File to store the TMalign program output""",
                 metavar="FILE")
    parser.add_option("-s", "--tm_alignment_file",
                 dest="tmalif",
                 type='string',
                 help="""File to store the TMalign alignment output,
                      modified for compatability with Modeller.""",
                 metavar="FILE")

    # Check mandatory options
    opts, args = parser.parse_args()

    if not opts.pdb1 or not opts.pdb2:
        parser.print_help()
        sys.exit(1)

    return opts, args

def calculate_tm_alignment(env, pdb1, chn1, pdb2, chn2,
                           tminpf, tmoutf, tmalif):
    """This does the actual work of calculating the TMalign alignment
    and parsing it. Also converts the TMalign output to a form that is
    compatible with Modeller."""

    # Create a TMAlign alignment obj and use it
    mytm = TMAlign(env, pdb1, chn1, pdb2, chn2)

    # Either use the given TMAlign alignment file or run TMalign
    # to get the alignment
    if tminpf:
        mytm.read_program_output(tminpf)
    else:
        mytm.get_alignment()
        if tmoutf:
            mytm.write_program_output(tmoutf)

    # Parse TMalign output and get the alignment
    mytm.parse_output()

    # Clean up the alignment
    mytm.fix_alignment()

    # Write out alignment if requested
    if tmalif:
        mytm.write_alignment(file=tmalif)

    # Create superpositions
    results = mytm.calculate_superpositions()

    del(mytm)
    return results


if __name__ == "__main__":

    # Parse command line options
    opts, args = main()

    # -- Initialize some modeller stuff
    log.level(1, 0, 0, 1, 0)
    env = Environ()
    env.io.atom_files_directory = opts.pdbrep

    # Call function to calculate TMalign alignments and parse them
    calculate_tm_alignment(env, opts.pdb1, opts.chn1,
                           opts.pdb2, opts.chn2, opts.tminpf,
                           opts.tmoutf, opts.tmalif)
