# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.


"""Simple cyclic redundancy check.
   This is based on the Perl implementation in SWISS::CRC64, which in turn is
   described by ISO 3309.
"""

import sys

# Initialisation
# 32 first bits of generator polynomial for CRC64
# the 32 lower bits are assumed to be zero
if sys.version_info[0] == 2:
    # long constants are a syntax error in python3
    _POLY64REVh = eval("0xd8000000L")
    ONE_LONG = long(1)
    EIGHT_LONG = long(8)
    THIRTY_ONE_LONG = long(31)
    ZERO_LONG = long(0)
else:
    _POLY64REVh = 0xd8000000
    ONE_LONG = 1
    EIGHT_LONG = 8
    THIRTY_ONE_LONG = 31
    ZERO_LONG = 0
_CRCTableh = [0] * 256
_CRCTablel = [0] * 256
_is_initialized = False

def digest(s):
    """Return the 64-bit CRC of a string, as a 16-byte hex-encoded string."""
    global _is_initialized
    crcl = 0
    crch = 0
    if not _is_initialized:
        _is_initialized = True
        for i in range(256):
            partl = i
            parth = ZERO_LONG
            for j in range(8):
                rflag = partl & ONE_LONG
                partl >>= ONE_LONG
                if (parth & 1):
                    partl |= (ONE_LONG << THIRTY_ONE_LONG)
                parth >>= ONE_LONG
                if rflag:
                    parth ^= _POLY64REVh
            _CRCTableh[i] = parth;
            _CRCTablel[i] = partl;

    for item in s:
        shr = ZERO_LONG
        shr = (crch & 0xFF) << 24
        temp1h = crch >> EIGHT_LONG
        temp1l = (crcl >> EIGHT_LONG) | shr
        tableindex = (crcl ^ ord(item)) & 0xFF

        crch = temp1h ^ _CRCTableh[tableindex]
        crcl = temp1l ^ _CRCTablel[tableindex]
    return "%08X%08X" % (crch, crcl)
