import shutil
from bc import Biocore, BioFSDirectory, BiocoreError
import os, os.path, string, sys
import dirmake
import tempfile
import chimera
from chimera import replyobj
import SimpleSession

## create a Biocore object that will be accessible to all functions in this module
b = Biocore("https://biocore-s.ks.uiuc.edu/biocore/servlet/",
                "DA267121866EBAB206793D672F5928A5", "153")

## create the GUIs used to Open/Save files to BioCoRe
import gui

## variable will track whether you are logged in
## (you are logged for 700 minutes after a call to b.login)
_alreadyLoggedIn = 0

## variable which tracks whether the local directory tree has been created
## to mirror the BioFS directory hierarchy.
_dirHierCalc    =  0

## path on the local file system, under which goes the mirrored copy of BioFS
## with all files of size 0. /tmp/BioCoRE is equilvalent to / on the BioFS
ROOTPATH = '/tmp/BioCoRE/'

def registerExtension(name, obj):
    import Extension
    Extension.extensions.append((name, obj))


def checkLogin():
    """Check to see if you're already logged in.
    If you're not already logged in, go ahead and do it.
    If you are, just return 1.
    Will return 0 if there was an error in the login process
    """

    global b
    global _alreadyLoggedIn

    if not _alreadyLoggedIn:
        replyobj.status("logging in to BioCoRE...")
        try:
            b.login("dan", "fubar")
        except BiocoreError, what:
            msg = "BioCoRE Error: %s" % what
            replyobj.status(msg, color='red')
            return 0
        else:
            _alreadyLoggedIn = 1
            ## register a logout function when Chimera exits...
            chimera.triggers.addHandler("Chimera exit", logout, None)
            return 1
    else:
        return 1

def logout(triggerName=None, closure=None, data=None):
    global b

    ## clean up temporary directory used to mirror BioFS
    try:
        shutil.rmtree(ROOTPATH)
    except IOError:
        pass

    try:
        replyobj.status("logging out of BioCoRE...")
        b.logout()
    except BiocoreError, what:
        replyobj.status(what, color='red')


def openSavePanel(mode):
    """This function is called by the ChimeraExtension mechanism.
    'mode' will either be one of 'open' or 'save'
    """

        
    global b
    global ROOTPATH
    global _dirHierCalc

    if not checkLogin():
        return

    ## if the local mirror of BioFS isn't already built
    if not _dirHierCalc:
        refreshDirHier()
        _dirHierCalc = 1
    
    if mode == 'open':
        chimera.dialogs.display('Open Session from BioCoRE')
    elif mode == 'save':
        processSaveRequest(ROOTPATH)



def processSaveRequest(initpath):
    ## eventually we will do something different if the user has already saved a
    ## session once, the name of the session will be in chimera.tkgui.lastSession.
    ## but for now, we can't delete a file from BioFS, so you wouldn't be able to
    ## just 'resave' (i.e. delete existing session file, save new session with
    ## same name. So for now, just pop up a dialog, must use a different name. 
    
    #if not hasattr(chimera.tkgui, 'lastSession'):
    #    m = SaveModeless(initialdir = initpath, command=processSaveSelection, filters=[("Chimera session", ["*.py"], ".py")])
    #else:
    #    processSaveSelection(path=chimera.tkgui.lastSession)
    
    chimera.dialogs.display('Save Session to BioCoRE')

def processSaveSelection(which=None, dialog=None, path=None):
    """Gets called when user selects a directory/file to save session to.
    which is 1 if 'OK' or 'Apply' was pressed, 0 if 'Cancel' was pressed.
    dialog is the actual dialog that was used.
    """

    if which == 0:
        ## Cancelled, don't do anything
        return
    
    if dialog:
        ## get list of paths that were picked
        paths = dialog.getPaths()
        
        if not paths:
            replyobj.warning("No save file selected, aborting save.\n")
            return
        ## can only save to one path
        path = paths[0]
        path = string.join(path.split(os.sep),'/')

    ## get just the file name
    sess_filename = os.path.split(path)[1]

    ## get the part of the file name before the '.'
    sess_prefix = sess_filename[0:string.find(sess_filename,'.')]
    
    SimpleSession.save.saveSession(path)

    ## parse the session file to get a string which is python code of a map
    ## where the keys contain the path of the coordinate files used in the session
    files_to_upload = parseSourceFiles(path)

    ## exec that string so you can actually access the variable srcMolMap
    exec files_to_upload

    ## so we know that we've already saved the session once
    ## remember, path is the path of the saved session file
    #chimera.tkgui.setLastSession(path)

    
    pathname,filename = os.path.split(path)
    pathname = string.join(pathname.split(os.sep),'/')

    ## so dirname is the name of the directory where the user chose
    ## to save the session
    dirname = os.path.split(pathname)[1]
    dirname = string.join(dirname.split(os.sep),'/')

    ## dirmake.dirmap is a map of {local directory names : BioFS IDs}
    ## look up dirname so we know the Id of the directory to upload the sesion file to
    try:
        dirId = dirmake.dirmap[dirname]
    except KeyError:
        replyobj.status("Error locating directory %s. Click 'Refresh' and try again" % dirname, color='red')
        return

    ## Actually upload the sesion file to the BioFS
    try:
        replyobj.status('saving session to BioCoRE....')
        b.putBiofsFile(dirId, path, filename)
    except BiocoreError, what:
        replyobj.status("Error saving session to BioCoRE: %s" % what , color='red', log=1)
    else:
        replyobj.status('saving session to BioCoRE....complete')

    ## now need to upload coordinate data files also...
    for key in srcMolMap.keys():
        ##srcPath is the path to the coordinate data file
        srcPath = key[0]
        srcPath = string.join(srcPath.split(os.sep),'/')
        

        ## if it has a '/' in it, then the file is located somewhere on the local system
        ## and cannot be fetched from the PDB. It must be uploaded
        if (string.find(srcPath,'/') >= 0) or (string.find(srcPath,'\\') >= 0):
            src_filename = os.path.split(srcPath)[1]
            try:
                replyobj.status("saving source file %s to BioCoRE...." % src_filename)

                ## save the file to BioCoRE, using the name CSS_sesionfilename_actualPDBfilename
                b.putBiofsFile(dirId, srcPath, "CSS_%s_%s" % (sess_prefix,src_filename) )
            except BiocoreError, what:
                replyobj.status("Error saving session to BioCoRE: %s" % what , color='red', log=1)
            else:
                replyobj.status("saving src file %s to BioCoRE....complete" % srcPath)
        



def parseSourceFiles(path):
    """parse the chimera session file located at path, and return a string that contains
    code which defines a map in which the keys contain the local paths to coordinate data files
    This function will have to change when simplesession changes.
    """
    
    f = open(path,'r')

    ## will contain the snippet of code
    srcs = ''

    ## whether currently parsing set of lines that defines target map
    in_srcs = 0

    while 1:
        line = f.readline()

        if not in_srcs:
            if string.find(string.lstrip(line),"srcMolMap") == 0:
                in_srcs = 1
                srcs = srcs + line
        else:
            if string.find(string.lstrip(line), "molOrder") == 0:
                break
            else:
                srcs = srcs + line

    return string.strip(srcs)
            
        
        

def processOpenSelection(which,dialog):
    """handler for when user selects session file to open."""

    global b

    ## will map the pathnames of source files stored in session file to their actual location
    ## on the local filesystem
    src_file_tmp_map = {}

    if which == 0:
        ## Cancel pressed
        return

    try:
        ## only open one file
        path = dialog.getPaths()[0]

        ## file is the session file to open (py)
        file = os.path.split(path)[1]
        
    except IndexError:
        pass
    else:
        ## get the BioCoRE ID associated with the chosen file
        try:
            id = dirmake.filemap[file]
        except KeyError:
             replyobj.status("Error locating %s. Click 'Refresh' and try again" % file, color='red')
             return

        ## make a temp file that will contain the session
        tmp = tempfile.mktemp('.py')
        tmp = string.join(tmp.split(os.sep),'/')

        ## actually get the sesion file from BioCoRE
        replyobj.status("fetching %s from BioCoRE...." % file)
        try:
            b.getBiofsFile(id, tmp)
        except BiocoreError, what:
            replyobj.status("BioCoRE Error: %s" % what, color='red')
            return
    
        replyobj.status("fetching %s from BioCoRE....complete" % file)
            
        ## now, need to also get the source coordinate data files (if any!)...
        files_to_download = parseSourceFiles(tmp)
        exec files_to_download
        
        for key in srcMolMap.keys():

            ## srcPath would be the path of source file on the saver's system
            srcPath = key[0]
            
            ## what to do when going from windows to other systems??
            if srcPath.find('\\') >= 0: ## session was saved on a windows system
                origSrcPath = srcPath
                srcPath = string.join(srcPath.split('\\'),'/')
            else:
                origSrcPath = srcPath
                
            ## if it has a '/' in it, it was on the saver's local FS, and got uploaded
            ## to BioCoRe (hopefully!!)
            if (string.find(srcPath,'/') >= 0) or (string.find(srcPath,'\\') >= 0): ##don't need second one..
                ## get just the source filename
                src_filename = os.path.split(srcPath)[1]
                ## sess_prefix is the part before the '.' of the session file
                sess_prefix = file[0:string.find(file,'.')]

                ## determine the proper filename to look for, and get the Id
                encoded_src_name = "CSS_%s_%s" % (sess_prefix, src_filename)

                try:
                    id = dirmake.filemap[encoded_src_name]
                except KeyError:
                     replyobj.status("Error locating source file %s. Click 'Refresh' and try again"  \
                                     % encoded_src_name, color='red')
                     return
                

                ## make a temp file to download the source file to
                src_tmp = tempfile.mktemp('.pdb')
                src_tmp = string.join(src_tmp.split(os.sep),'/')
                

                ## use the path of the other one, but name the file the original pdb file name
                renamed_src_tmp = os.path.join(os.path.split(src_tmp)[0], src_filename)
                renamed_src_tmp = string.join(renamed_src_tmp.split(os.sep),'/')

                                    
                try:
                    b.getBiofsFile(id, renamed_src_tmp)
                    ## establish a mapping btn. the path of the pdb file specified in the session file
                    ## and the path of the pdb file on the local system
                    src_file_tmp_map[origSrcPath] = renamed_src_tmp
                except BiocoreError, what:
                    replyobj.status("BioCoRE Error: %s" % what, color='red')
                    return

        ## Need to post-process the session file, substituting the path of source files on the local system
        ## in place of the path of the src file on the saver's system (which is written into the session file)
        processedSess = tempfile.mktemp('.py')
        processedSess = string.join(processedSess.split(os.sep),'/')

        ## read from the session file
        readfrom = open(tmp,'r')
        ## write to the newly-created 'processedSess' file
        writeto = open(processedSess, 'w')
            
        for s in readfrom.xreadlines():
            for k,v in src_file_tmp_map.items():
                s = s.replace(repr(k),"'%s'" % v)
            writeto.write(s)

        readfrom.close()
        writeto.close()
            
        try:
            chimera.openModels.open(processedSess) ##was tmp
        except:
            replyobj.status("Cannot open file %s" % file, color='red')


def createNewFolder(new_folder_path, new_folder):
    global b

    newDirParent = os.path.split(os.path.split(new_folder_path)[0])[1]
    b.createBiofsDirectory(dirmake.dirmap[newDirParent], new_folder)


def refreshDirHier(startfrom=None):
    """ startfrom is the path of the folder you want to start recursively
    refreshing from (e.g. the folder above the a new folder you just created)
    """
    dirs = []
    files = []

    global b

    ## /tmp/BioCoRE (local) == / (BioFS)
    ## let TOP be the directory you want to recursively refresh..

    if startfrom:
        TOP = startfrom
    else:
        TOP = ROOTPATH


    ## need to get the right id
    if TOP == ROOTPATH:
        id = 0
    else:
        id = dirmake.dirmap[os.path.split(startfrom)[1]]

    if os.access(TOP, os.F_OK):
        shutil.rmtree(TOP)
    os.makedirs(TOP)

    ## put the id file back in there...
    o = open(os.path.join(TOP,".myIdNumber"), 'w')
    o.write("%s\n" % id)
    o.close()

    ## at this point, files exist on BioFS but have been deleted from local system
    try:
        dirs,files = b.listBiofsDirContents(id)
    except BiocoreError, what:
        msg = "BioCoRE Error: %s" % what
        replyobj.status(msg, color='red')
        return

    ## rebuild internal structures that map files/directories to their ID numbers
    ## (dirmake.dirmap, dirmake.filemap)
    dirmake.buildMaps(ROOTPATH)

    ids = dirmake.dirmap.values()
    
    replyobj.status('retrieving information from BioCoRE.....')
    try:
        dirmake.dirmake(ids, TOP, dirs, files, b)
    except BiocoreError, what:
        replyobj.status("BioCoRE Error: %s" % what, color='red')
        return 0
    else:
        replyobj.status('retrieving information from BioCoRE.....complete')
        return 1


