#***************************************************************************
# 
#            (C) Copyright 1995-2003 The Board of Trustees of the
#                        University of Illinois
#                         All Rights Reserved
#
#***************************************************************************

# biocore API in TCL
# usage:  In tcl, do a
#     source biocore.tcl
# and then you are in good shape (or, you should be able to put this in
# a package directory and be OK)

# Change Log
# 1.01:
#   login proc added
# 1.02:
#   get project list proc added
# 1.03:
#   is user currently valid proc added.
# 1.04:
#   file download proc added
# 1.05:
#   file upload proc added
# 1.06:
#   logout proc added
# 1.07:
#   biofs del file, del directory, and create dir procs added

#  future:
#   send cp chat proc added


global BIOCORE_API_VERSION
set BIOCORE_API_VERSION 1.07

package provide biocore $BIOCORE_API_VERSION

namespace eval ::biocore:: {


}

if {![info exists biocoreHush ]} {
   puts "BioCoRE (http://www.ks.uiuc.edu/Research/biocore/) API Version $BIOCORE_API_VERSION"
}

package require http 2.0

global hasTLS
set hasTLS 1

if [catch {
package require tls 
http::register https 443 ::tls::socket
} ] {
   set hasTLS 0
   if {![info exists biocoreHush ]} {
      puts "HTTPS not available. (no TLS package)  Connections must be made via HTTP."
   }
}


# ----------------------------------  =  --------------------------------------
# -------------- standard code to give an object - oriented feel
# -------------- to what we are doing

# 'delete' procedure independent of the class
proc delete {args} {
   foreach name $args {
      upvar #0 $name arr
      unset arr        ; # Deletes the object's data
      rename $name {}  ; # Deletes the object command
   }
} ; # end of delete

# No more 'methods' argument here; 'vars' is optional
proc class {classname {vars ""}} {

   # Create the class command, which will allow new instances to be created.
   set template {
      proc @classname@ {obj_name args} {
         # The class command in turn creates an object command.
         # Fewer escape characters thanks to the '@' sign.
         proc $obj_name {command args} \
            "return \[eval dispatch_@classname@ $obj_name \$command \$args\]"

         # Set variable defaults, if any
         upvar #0 $obj_name arr
         @set_vars@

         # Then possibly override those defaults with user-supplied values
         if { [llength $args] > 0 } {
            eval $obj_name configure $args
         }
      }
   }

   set set_vars "array set arr {$vars}"
   regsub -all @classname@ $template $classname template
   if { $vars != "" } {
      regsub -all @set_vars@  $template $set_vars template
   } else {
      regsub -all @set_vars@  $template "" template
   }

   eval $template

   # Create the dispatcher, which does not check what it
   # dispatches to.
   set template {
      proc dispatch_@classname@ {obj_name command args} {
         upvar #0 $obj_name arr
         if { $command == "configure" || $command == "config" } {
            array set arr $args

         } elseif { $command == "cget" } {
            return $arr([lindex $args 0])

         } elseif { $command == "toList" } {
            return [array get arr]

         } else {
            uplevel 1 @classname@_${command} $obj_name $args
         }
      }
   }

   regsub -all @classname@ $template $classname template

   eval $template
} ; # end of class

# ----------------------------------  =  --------------------------------------
# ----------------------------------  =  --------------------------------------
# ----------------------------------  =  --------------------------------------
# ----------------------------------  =  --------------------------------------
# ----------------------------------  =  --------------------------------------
# ----------------------------------  =  --------------------------------------
# ----------------------------------  =  --------------------------------------
# helper URL classes
proc addIdentParms { strSess strProj otherParam } {
   global BIOCORE_API_VERSION

   upvar $otherParam params

   # add in our api commands to tell the server that the API is
   # requesting this
   set params(src) "api"
   set params(apiv) $BIOCORE_API_VERSION

   # set the session if one was given
   if { [string compare $strSess ""] != 0 } {
      set params(jsessionid) $strSess
   }
   # set the project if one was given
   if { [string compare $strProj ""] != 0 } {
      set params(p) $strProj
   }

   return [array get params]
} ; # end of addIdentparms

proc encodeURLParams { strSess strProj otherParam } {
   upvar $otherParam pList

   set params [addIdentParms $strSess $strProj pList]

#   return [eval ::http::formatQuery [array get params]]
   return [eval ::http::formatQuery $params]

} ; # end of encodeURLParams


# ----------------------------------  =  --------------------------------------
proc encodeAndSendFile {strURL strSess strProj strServ par binFileData \
                                strDataVar strRemoteName } {
   package require http 2.2

   upvar $par otherParams

   # start preparing the query
   set binQuery {}
   set strDivider "-------------------[clock seconds][pid]"

   #append binQuery "--"

   append binQuery "--${strDivider}\r\nContent-Disposition: form-data;\
            name=\"$strDataVar\"; filename=\"$strRemoteName\"\r\n\r\n$binFileData\r\n"

   set parList [addIdentParms $strSess $strProj otherParams]

   # parList is now an array of the rest of the choices
   foreach {elem data} $parList {
        append binQuery "--$strDivider\r\nContent-Disposition: form-data;\
                name=\"$elem\"\r\n\r\n$data\r\n"   
   }

   append binQuery "--${strDivider}--"

   # now the query 'string' is ready to go.
   # puts "$binQuery"

   set httpHandle [::http::geturl $strURL$strServ \
            -type "multipart/form-data; boundary=$strDivider" \
            -query $binQuery] 
   
   return [::http::data $httpHandle]

} ; # end of encodeAndSendFile

# ----------------------------------  =  --------------------------------------
proc encodeAndSend {strMethodType strURL strSess strProj strServ par} {

   # have to use upvar to pass an array to a procedure
   upvar $par otherParam

   set strParamList [encodeURLParams $strSess $strProj otherParam]

   # test whether this is a GET or a POST
   if { [string match $strMethodType "get"] } {
      set strFullUrl "$strURL$strServ?$strParamList"
      #puts "requesting: $strFullUrl"
      set httpHandle  [::http::geturl $strFullUrl ]
   } else {
      set strFullUrl "$strURL$strServ"
      set httpHandle  [::http::geturl $strFullUrl -query $strParamList]
   }
   
   return [::http::data $httpHandle]

} ; # end of encodeAndSend

# ----------------------------------  
proc isValidBiocoreResponse { strResponse listR iSVer} {
   upvar $listR listResult
   upvar $iSVer iServerVersion

   set listSplit [ split $strResponse "\n" ]

   # first value sent back is the server version
   set iServerVersion [lindex $listSplit 0]

#   puts "iserver is $iServerVersion"
   # second value is a true/false
   set strValid [lindex $listSplit 1]

   # prepare the list for future use
   set listResult [ lrange $listSplit 2 end]

#   puts "strvalid is $strValid"
   return [string match $strValid "true"]
} ; # end of isValidBiocoreResponse

# ----------------------------------  
proc biocoreHttpCopy {url file {chunk 4096} } {
  set out [open $file w]
  set token [::http::geturl $url -channel $out -blocksize $chunk]
  close $out
  # This ends the line started by http::Progress
  upvar #0 $token state
  set max 0
  foreach {name value} $state(meta) {
    if {[string length $name] > $max} {
      set max [string length $name]
    }
  }
#  foreach {name value} $state(meta) {
#    puts [format "%-*s %s" $max $name: $value]
#  }
}  


# ----------------------------------  =  --------------------------------------
# ----------------------------------  =  --------------------------------------
# ----------------------------------  
# -------- Biocore "classes"
class UserInfo

class NotebookEntry {Reply "" Description ""}
proc NotebookEntry_print {self} {
   puts "Type: [$self cget Type], Id: '[$self cget Id]', Time: '[$self cget Time]'"
   puts "User: [$self cget User], Title: '[$self cget Title]'\n[$self cget Text]"
}

class NotebookEntryInfo 

class ControlPanelChat

class BiofsDirectory


class BiofsDirectoryEntry
proc BiofsDirectoryEntry_printDir {self} {
   puts "DIR Id: [$self cget Id], Par: '[$self cget Parent]', Name: '[$self cget Name]', Icon: '[$self cget Icon]', Time: '[$self cget Time]'"
}
# ----
proc BiofsDirectoryEntry_printFile {self} {
   puts "FILE Id: [$self cget Id], Name: '[$self cget Name]',\nURL: '[$self cget URL]',\nIcon: '[$self cget Icon]', Length: [$self cget Length], User: '[$self cget User]', Time: '[$self cget Time]'"
}
# ----
proc BiofsDirectoryEntry_scanDir {self iCurLine listResult} {
   # the '2' was discovered after much trial and error.  Because this
   # method is hidden as part of a class, simply saying upvar $a b
   # doesn't work.  The calling method is actually 2 layers up the
   # stack
   upvar 2 $iCurLine iCurrentLine

   # first thing we get back now is the id of the directory
   $self configure Id [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back whether or not this is a 'parent'
   $self configure Parent [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the name of this directory entry
   $self configure Name [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the url of an icon for this directory entry
   $self configure Icon [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the time of this directory entry
   $self configure Time [lindex $listResult $iCurrentLine]
   incr iCurrentLine
} ; # end of BiofsDirectoryEntry_scanDir

# ----
proc BiofsDirectoryEntry_scanFile {self iCurLine listResult} {
   # see scanDir for an explanation of the '2'
   upvar 2 $iCurLine iCurrentLine

   # first thing we get back now is the id of the file
   $self configure Id [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the name of this file
   $self configure Name [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the URL of this file
   $self configure URL [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the URL of an icon for this file
   $self configure Icon [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the Length of this file
   $self configure Length [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the User of this file
   $self configure User [lindex $listResult $iCurrentLine]
   incr iCurrentLine

   # next we get back the time of this file
   $self configure Time [lindex $listResult $iCurrentLine]
   incr iCurrentLine
} ; # end BiofsDirectoryEntry_scanFile

class Biocore
# ----------------------------
# set proxy information
proc Biocore_setProxy {self strHost iPort} {
   ::http::config -proxyhost $strHost -proxyport $iPort
} ; # end of setProxyInformation

# -----
# saving the notebook entry
proc Biocore_saveNotebookEntry {self ne} {
   set nbparams(subject) [ne cget Title]
   set nbparams(description) [ne cget Description]
   set nbparams(data) [ne cget Text]
   set nbparams(msgType) [ne cget Type]

   if { [string compare [ne cget Reply] ""] != 0 } {
      set nbparams(reply) [ne cget Reply]
   }

   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend post [$self cget URL] [$self cget Session] \
                       [$self cget Project] "NoteBook" nbparams]

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server
      set iSuccess [ lindex $listResult 0 ]
   } else {
      set iSuccess -2
   }

   return $iSuccess
} ; # end of Biocore_saveNotebookEntry

# ----------------------------
# get a list of notebook entries
proc Biocore_getNotebookEntryList {self iType iStart iEnd} {
   set nbparams(msgType) $iType
   set nbparams(start) $iStart
   set nbparams(end) $iEnd
   
   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend get [$self cget URL] [$self cget Session] \
                       [$self cget Project] "NoteBook" nbparams]

   #puts "I got\n---\n$strResponse\n---"

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server

      # set the line in the listResult that we are currently reading
      set iCurrentLine 0      

      # first thing we get back now is the number of notebook entries
      set iNumEntries [lindex $listResult $iCurrentLine]
      incr iCurrentLine

      # make ourselves a list for the results
      set listNotebookEntries [list]

      for {set iCount 0} {$iCount < $iNumEntries} {incr iCount} {
         NotebookEntryInfo nei

         # first we get the ID
         nei configure Id [lindex $listResult $iCurrentLine]
         incr iCurrentLine

         # now we have to test to see if this is a 'deleted' message
         if {[nei cget Id] != -1} {
            # get the time
            nei configure Time [lindex $listResult $iCurrentLine]
            incr iCurrentLine

            # get the user
            nei configure User [lindex $listResult $iCurrentLine]
            incr iCurrentLine

            # get the title
            nei configure Title [lindex $listResult $iCurrentLine]
            incr iCurrentLine

            # get the description
            nei configure Description [lindex $listResult $iCurrentLine]
            incr iCurrentLine
         } ; # end the if on us not having a deleted message

         # add this notebook entry to the list
         lappend listNotebookEntries [nei toList]
      } ; # end the for loop over the notebook entries
   } else {
      error "Invalid User."
   } ; # end the else on the user not being valid

   return $listNotebookEntries
} ; # end of Biocore_getNotebookEntryList 
# ----------------------------
# get a single notebook entry
proc Biocore_getNotebookEntry {self iMsgId} {
   NotebookEntry ne
   set nbparams(show) $iMsgId
   
   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend get [$self cget URL] [$self cget Session] \
                       [$self cget Project] "NoteBook" nbparams]

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server

      # set the line in the listResult that we are currently reading
      set iCurrentLine 0      

      # first thing we get back now is whether or not the request was valid
      set iSuccess [lindex $listResult $iCurrentLine]
      incr iCurrentLine

      if { $iSuccess == 0} {
         error "Unauthorized access"
      } else {
         # first we get the Type
         ne configure Type [lindex $listResult $iCurrentLine]
         incr iCurrentLine

         # Next we get the Id
         ne configure Id [lindex $listResult $iCurrentLine]
         incr iCurrentLine

         # Next we get the Title
         ne configure Title [lindex $listResult $iCurrentLine]
         incr iCurrentLine

         # Next we get the User
         ne configure User [lindex $listResult $iCurrentLine]
         incr iCurrentLine

         # Next we get the Description
         ne configure Description [lindex $listResult $iCurrentLine]
         incr iCurrentLine

         # Next we get the Time
         ne configure Time [lindex $listResult $iCurrentLine]
         incr iCurrentLine

         # Lastly we get the actual Text
         ne configure Text [join [lrange $listResult $iCurrentLine end] "\n"]

      } ; # end the else on the user being authorized to see entry
   } else {
      error "Invalid User."
   }

   return [ne toList]
} ; # end of Biocore_getNotebookEntry 
# ----------------------------
# login to BioCoRE
proc Biocore_login {self strName strPass} {
   global hasTLS 
   # require TLS
   if { $hasTLS == 0 } {
      # No secure sockets..  Login not allowed
      # comment out the next line to allow the client to try talking to server
      error "No TLS.  Login not allowed"
   }

   # We are using TLS
   set logparams(uname) $strName
   set logparams(pass)  $strPass

   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend post [$self cget URL] 0 \
                       0 "CheckLogin" logparams]

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      UserInfo ui

      # listResult is now a list of the remainder of the response from the
      # server
      $self configure Session [ lindex $listResult 0 ]

      ui configure Id [ lindex $listResult 1 ]
   } else {
      error "Invalid User."
   }

   return [ui toList]
} ; 

# ----------------------------
# logout of BioCoRE
proc Biocore_logout {self} {
   set parms(logout) "logout"
   set parms(noRedirect) "true"

   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend get [$self cget URL] [$self cget Session] \
                       [$self cget Project] "NewMain" parms]


   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      return 1

   } else {
      # Shouldn't ever get here.  Logging out will just fail silently if
      # the user isn't already logged in
      error "Invalid User."
   }

   return 0
} ;  # end of Biocore_logout
# ----------------------------
# Is this a valid user?
proc Biocore_isUserValid {self} {
   
   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend get [$self cget URL] [$self cget Session] \
                       [$self cget Project] "Main" ""]

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server, which shouldn't be anything
      return 1
   } else {
      error "Invalid User."
   }

   return 0
} ; # end of Biocore_isUserValid 

# ----------------------------------  =  --------------------------------------
#################
# 
# BioCore_getProjectList
#
# gets a list of all the projects a user is in
# 
# returns an list in the form of [Name1 Id1 Name2 Id2... ]
proc Biocore_getProjectList { self } {
    #puts "starting BioCoRE_getProjectList"
    set fsparams(utilDest) "getProjectList"
    set projList ""

    set strResponse [encodeAndSend get [$self cget URL] [$self cget Session] [$self cget Project] "Utilities" fsparams]
    if { [ isValidBiocoreResponse $strResponse listResult iServerVersion] } {
	regsub "{" $listResult "" listResult
	regsub "}" $listResult "" listResult
	regsub -all ";" $listResult "," listResult
	set projList [split $listResult ,]
    } else {
       error "Invalid request."
    }

    return $projList
}

# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------

# ----------------------------------  =  --------------------------------------
# BioCore_getBioFSFile
#
# save a file from the biofs onto the local filesystem
# 
proc Biocore_getBiofsFile  { self biofsID strFileLocation } {
   set params(dest) "get"
   set params(type) "f"
   set params(id) $biofsID

   set strParamList [encodeURLParams [$self cget Session] [$self cget Project] params]

   set strFullURL "[$self cget URL]Fs?$strParamList"

   biocoreHttpCopy $strFullURL $strFileLocation
}

# ----------------------------------  =  --------------------------------------
# BioCore_delBiofsFile
#
# Delete a file from the BioFS
proc Biocore_delBiofsFile {self biofsID} {
   set params(dest) "del"
   set params(type) "f"
   set params(id) $biofsID

   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend post [$self cget URL] [$self cget Session] \
                       [$self cget Project] "Fs" params]

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server
      # set the line in the listResult that we are currently reading
      set iCurrentLine 0      

      # first thing we get back now is whether or not the request was valid
      set iSuccess [lindex $listResult $iCurrentLine]
      incr iCurrentLine

      if { $iSuccess == 0} {
         # an error message is also stored.  So, let's return that also
         error "[lindex $listResult $iCurrentLine]"
      } 
   } else {
      error "Invalid User."
   }

} ; # end of Biocore_delBiofsFile


# ----------------------------------  =  --------------------------------------
# BioCore_delBiofsDirectory
#
# Delete a Directory from the BioFS
proc Biocore_delBiofsDirectory {self biofsID} {
   set params(dest) "del"
   set params(type) "d"
   set params(id) $biofsID

   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend post [$self cget URL] [$self cget Session] \
                       [$self cget Project] "Fs" params]

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server
      # set the line in the listResult that we are currently reading
      set iCurrentLine 0      

      # first thing we get back now is whether or not the request was valid
      set iSuccess [lindex $listResult $iCurrentLine]
      incr iCurrentLine

      if { $iSuccess == 0} {
         # an error message is also stored.  So, let's return that also
         error "[lindex $listResult $iCurrentLine]"
      } 
   } else {
      error "Invalid User."
   }

} ; # end of Biocore_delBiofsDirectory


# ----------------------------------  =  --------------------------------------
# BioCore_delBiofsDirectory
#
# Delete a Directory from the BioFS
proc Biocore_createBiofsDirectory {self biofsID strName} {
   set params(dest) "create"
   set params(type) "d"
   set params(id) $biofsID
   set params(dirName) $strName

   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend post [$self cget URL] [$self cget Session] \
                       [$self cget Project] "Fs" params]

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server
      # set the line in the listResult that we are currently reading
      set iCurrentLine 0      

      # first thing we get back now is whether or not the request was valid
      set iSuccess [lindex $listResult $iCurrentLine]
      incr iCurrentLine

      if { $iSuccess == 0} {
         # an error message is also stored.  So, let's return that also
         error "[lindex $listResult $iCurrentLine]"
      } 
   } else {
      error "Invalid User."
   }

} ; # end of Biocore_createBiofsDirectory


# ----------------------------------  =  --------------------------------------
# BioCore_putBioFSFile
#
# save a file from the local filesystem into the Biofs
# 
proc Biocore_putBiofsFile  { self biofsDirID strLocal strRemote } {

    set iId -1
   # load the file into   $binData   ....
   set fid [open $strLocal r]
   fconfigure $fid -translation binary
   if {[catch {read $fid [file size $strLocal]} binData]} {
      return -code error $binData
   }
   close $fid

   set params(id) $biofsDirID
   set params(dest) "put"
   set params(desiredFileName) $strRemote

   # we are now ready to actually do the URL work
   set strResponse [encodeAndSendFile [$self cget URL] [$self cget Session] \
                       [$self cget Project] "Fs" params $binData \
                       "upload_file" $strRemote ]
   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server

      # set the line in the listResult that we are currently reading
      set iCurrentLine 0      

      # first thing we get back now is whether or not the request was valid
      set iSuccess [lindex $listResult $iCurrentLine]
       set iId [lindex $listResult 1]
      incr iCurrentLine

      if { $iSuccess == 0} {
         # an error message is also stored.  So, let's tell the user
         error "[lindex $listResult $iCurrentLine]"
      } 
   } else {
      error "Invalid request.\n$listResult"
   }

    return $iId
   # return $iSuccess
} ; # end of Biocore_putBiofsFile

# ----------------------------
# get a biofs listing
proc Biocore_getBiofsDirectory {self iId iLimit} {
   BiofsDirectory bioDir
   set fsparams(dest) "get"
   set fsparams(id) $iId
   if { $iLimit != 0 } {
      set fsparams(projlimit) "true"
   }  
 
   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend get [$self cget URL] [$self cget Session] \
                       [$self cget Project] "Fs" fsparams]

   #puts "I got\n---\n$strResponse\n---"

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server

      # set the line in the listResult that we are currently reading
      set iCurrentLine 0      

      # first thing we get back now is whether or not the request was valid
      set iSuccess [lindex $listResult $iCurrentLine]
      incr iCurrentLine

      if { $iSuccess == 0} {
         # an error message is also stored.  So, let's return that also
         error "Unauthorized access: [lindex $listResult $iCurrentLine]"
      } else {
         # first thing we get back now is the name of the directory
         bioDir configure Name [lindex $listResult $iCurrentLine]
         incr iCurrentLine

         # next, we get a number of directory entries
         set iNumEntries [lindex $listResult $iCurrentLine]
         incr iCurrentLine
   
         # make ourselves a list for the results
         set listEnt [list]

         for {set iCount 0} {$iCount < $iNumEntries} {incr iCount} {
            BiofsDirectoryEntry bioDE
   
            # first we get the type of this entry
            bioDE configure Type [lindex $listResult $iCurrentLine]
            incr iCurrentLine
  
            # test for whether it is a directory or a file
            if { [bioDE cget Type] == 0 } {
               bioDE scanDir iCurrentLine $listResult
            } else {
               bioDE scanFile iCurrentLine $listResult
            } ; # end the else
   
            # add this notebook entry to the list
            lappend listEnt [bioDE toList]
         } ; # end the for loop over the notebook entries
   
         bioDir configure Entries $listEnt
      } ; # end the else on this being a valid access
   } else {
      error "Invalid User."
   } ; # end the else on the user not being valid

   return [bioDir toList]
} ; # end of Biocore_getBiofsDirectory 


# ----------------------------------  =  --------------------------------------
# -----
# add a chat to the Control Panel
proc Biocore_saveCPChat {self cpc} {
   set cpparams(message) [cpc cget Msg]
   set cpparams(send) [cpc cget Send]
   set cpparams(cpid) "-1"

   if { [string compare [cpc cget Send] "privateChat"] == 0 } {
      set cpparams(recip) [cpc cget Recip]
   }

   # we are now ready to actually do the URL work
   set strResponse [encodeAndSend post [$self cget URL] [$self cget Session] \
                       [$self cget Project] "NewMain" cpparams]

   # is this user valid?
   if { [ isValidBiocoreResponse $strResponse listResult iServerVersion ] } {
      # listResult is now a list of the remainder of the response from the
      # server
      set iSuccess [ lindex $listResult 0 ]
   } else {
      set iSuccess -2
   }

   return $iSuccess
} ; # end of Biocore_saveCPChat

# ----------------------------------  =  --------------------------------------
