c --- This program will calculate the protrusion index as defined in
c     J.M. Thornton, M.S. Edwards, W.R. Taylor and D.J. Barlow,
c     Location of 'continuous' antigenic determinants in the
c     protruding regions of proteins. EMBO J. 2, 409-413, 1986.
c
c --- Compile by (most of routines in files geom1.f ... are not needed):
c
c     f77 ellipse.f geom1.f io.f matrix.f strnum2.f orthog.f -o ellipse
c     
c     5/2011: UP: bugfix in declaration (maxres was declared again after setting the parameter)

      program ellipse
        implicit none
        integer maxres,ioinp,ioout
        parameter (maxres = 1000, ioinp=11, ioout=12)
        integer nres,ierr,iargc,i
        real x(maxres),y(maxres),z(maxres),drmsq1,pi(maxres)
        real x2(maxres),y2(maxres),z2(maxres),a,b,c
        character resnam(maxres)*(3)
        character brkname*(100), piname*(100)

        if (iargc() .ne. 1) stop 'usage: ellipse pdb_file_name'
        call getarg(1, brkname)
        call rootfn(brkname, piname)
        call addstr(piname, '.pi')

        call rdca(ioinp,brkname,x,y,z,maxres,nres,resnam,ierr)
        if (ierr .gt. 0)
     -     stop 'Error[ellipse]; openning/reading PDB file'

        do  i = 1, nres
          x2(i) = x(i)
          y2(i) = y(i)
          z2(i) = z(i)
        end do

c ----- calculate the 'equivalent' ellipsoid and rotate the structure in
c       such a way that the momenta axes will correspond to the

c        call mmomrot(x2,y2,z2,nres,a,b,c)
c        write(*,*) 'Checking transformation; DRMS (should be ~0): ',
c     -             drmsq1(x,y,z,x2,y2,z2,nres)
c        write(*,'(a,3f8.3)') 'a,b,c   : ', a,b,c
c        write(*,'(a,3f8.3)') 'b/a, c/a: ', b/a,c/a

        call emomrot(x2,y2,z2,nres,a,b,c)
        write(*,*) 'Checking transformation; DRMS (should be ~0): ',
     -             drmsq1(x,y,z,x2,y2,z2,nres)
        write(*,'(a,3f8.3)') 'a,b,c   : ', a,b,c
        write(*,'(a,3f8.3)') 'b/a, c/a: ', b/a,c/a

        call getpi(x2,y2,z2,nres,a,b,c,pi)

        call openf(ioout, piname, 'unknown')
        do  i = 1, nres
          write(ioout, '(i5,1x,f5.1)') i, pi(i)
        end do
        close(ioout)

        stop
      end


c --- This routine will rotate the object in x,y,z to superpose the
c     three inertia axes with the coordinate system axes and also
c     translate its gravity centre to the origin of the coord. system.
c
c --- Produces the axes og the `equivalent' ellipsoid; better describe
c     the shape of the object than the inertia ellipsoid.
c
c --- The orientation is very similar to the inertia ellipsoid.
c
c --- Needs: ~/lib/matrix.f and ~/lib/orthog.f

      subroutine emomrot(x,y,z,n,a,b,c)
        implicit none
        integer n
        real x(n), y(n), z(n), rmom(3,3), eval(3), evec(3,3), detrm
        real gx,gy,gz,a,b,c

c ----- shift the molecule's gravity center to origin:
        call centr5(x, y, z, n, gx, gy, gz)

c ----- get the 2nd moment matrix (x,y,z gravity center must be shifted
c       to origin!!)
        call moment2(x,y,z,n,rmom)

c ----- get normalized and sorted eigenvectors of the inertia matrix
        call meigns(rmom,eval,evec,3,3)
        a = sqrt(eval(1))
        b = sqrt(eval(2))
        c = sqrt(eval(3))

c ----- if it happens that the coordinate system spanned by the eigenvectors
c       is lefthanded, convert it to the righthanded one by changing the
c       direction of the first eigenvector:
        detrm=evec(1,1)*evec(2,2)*evec(3,3)+evec(2,1)*evec(3,2)*
     -                                                evec(1,3)+
     -        evec(1,2)*evec(2,3)*evec(3,1)-evec(3,1)*evec(2,2)*
     -                                                evec(1,3)-
     -        evec(2,1)*evec(1,2)*evec(3,3)-evec(3,2)*evec(2,3)*
     -                                                evec(1,1)
        if (detrm .lt. 0.0) then
          evec(1,1) = -evec(1,1)
          evec(2,1) = -evec(2,1)
          evec(3,1) = -evec(3,1)
        end if

c ----- invert (transpose) the eigenvectors matrix to get the rotation
c       matrix:
        call rswap(evec(1,2), evec(2,1))
        call rswap(evec(1,3), evec(3,1))
        call rswap(evec(2,3), evec(3,2))

c ----- test and renormalize the rotation matrix
        call orthog(evec)

c        write(*,*) 'ROTATION MAT: '
c        do  i = 1, 3
c          write(*,*) (evec(i,j),j=1,3)
c        end do
c        write(*,*)

c ----- apply the rotation
        call rotat3(x,y,z,n,evec)

        return
      end



c --- This routine will rotate the object in x,y,z to superpose the
c     three inertia axes with the coordinate system axes and also
c     translate its gravity centre to the origin of the coord. system.
c
c --- Produces the axes of the inertia ellipsoid.
c
c --- Needs: ~/lib/matrix.f and ~/lib/orthog.f

      subroutine mmomrot(x,y,z,n,a,b,c)
        implicit none
        integer n
        real x(n), y(n), z(n), rmom(3,3), eval(3), evec(3,3), detrm
        real gx,gy,gz,a,b,c

c ----- shift the molecule's gravity center to origin:
        call centr5(x, y, z, n, gx, gy, gz)

c ----- get the inertia matrix (x,y,z gravity center must be shifted
c       to origin!!)
        call inertia(x,y,z,n,rmom)

c ----- get normalized and sorted eigenvectors of the inertia matrix
        call meigns(rmom,eval,evec,3,3)
        a = 1.0/sqrt(eval(3))
        b = 1.0/sqrt(eval(2))
        c = 1.0/sqrt(eval(1))

c ----- if it happens that the coordinate system spanned by the eigenvectors
c       is lefthanded, convert it to the righthanded one by changing the
c       direction of the first eigenvector:
        detrm=evec(1,1)*evec(2,2)*evec(3,3)+evec(2,1)*evec(3,2)*
     -                                                evec(1,3)+
     -        evec(1,2)*evec(2,3)*evec(3,1)-evec(3,1)*evec(2,2)*
     -                                                evec(1,3)-
     -        evec(2,1)*evec(1,2)*evec(3,3)-evec(3,2)*evec(2,3)*
     -                                                evec(1,1)
        if (detrm .lt. 0.0) then
          evec(1,1) = -evec(1,1)
          evec(2,1) = -evec(2,1)
          evec(3,1) = -evec(3,1)
        end if

c ----- invert (transpose) the eigenvectors matrix to get the rotation
c       matrix:


c ----- get normalized and sorted eigenvectors of the inertia matrix
        call meigns(rmom,eval,evec,3,3)
        a = 1.0/sqrt(eval(3))
        b = 1.0/sqrt(eval(2))
        c = 1.0/sqrt(eval(1))

c ----- if it happens that the coordinate system spanned by the eigenvectors
c       is lefthanded, convert it to the righthanded one by changing the
c       direction of the first eigenvector:
        detrm=evec(1,1)*evec(2,2)*evec(3,3)+evec(2,1)*evec(3,2)*
     -                                                evec(1,3)+
     -        evec(1,2)*evec(2,3)*evec(3,1)-evec(3,1)*evec(2,2)*
     -                                                evec(1,3)-
     -        evec(2,1)*evec(1,2)*evec(3,3)-evec(3,2)*evec(2,3)*
     -                                                evec(1,1)
        if (detrm .lt. 0.0) then
          evec(1,1) = -evec(1,1)
          evec(2,1) = -evec(2,1)
          evec(3,1) = -evec(3,1)
        end if

c ----- invert (transpose) the eigenvectors matrix to get the rotation
c       matrix:
        call rswap(evec(1,2), evec(2,1))
        call rswap(evec(1,3), evec(3,1))
        call rswap(evec(2,3), evec(3,2))

c ----- test and renormalize the rotation matrix
        call orthog(evec)

c        write(*,*) 'ROTATION MAT: '
c        do  i = 1, 3
c          write(*,*) (evec(i,j),j=1,3)
c        end do
c        write(*,*)

c ----- apply the rotation
        call rotat3(x,y,z,n,evec)

        return
      end



      subroutine rotat3(x, y, z, n, rotm)
      implicit none
      integer i, n
      real x(n), y(n), z(n), b1, b2, b3, rotm(3,3)
      do 10  i = 1, n
        b1=rotm(1,1)*x(i)+rotm(1,2)*y(i)+rotm(1,3)*z(i)
        b2=rotm(2,1)*x(i)+rotm(2,2)*y(i)+rotm(2,3)*z(i)
        b3=rotm(3,1)*x(i)+rotm(3,2)*y(i)+rotm(3,3)*z(i)
        x(i) = b1
        y(i) = b2
        z(i) = b3
10    continue
      return
      end


      subroutine getpi(x,y,z,n,a,b,c,pi)
        implicit none
        integer maxn
        parameter (maxn=1000)
        integer i,n,j,imin,indx(maxn)
        real x(n),y(n),z(n),pi(n),a,b,c,a2,b2,c2

        a2 = a*a
        b2 = b*b
        c2 = c*c
        do  i = 1, n
          pi(i) = x(i)**2/a2 + y(i)**2/b2 + z(i)**2/c2
          indx(i) = i
        end do

        do  i = 1, n-1
          imin = i
          do  j = i+1, n
            if (pi(j) .lt. pi(imin)) imin = j
          end do
          call swap(pi(i),pi(imin))
          call iswap(indx(i),indx(imin))
        end do

        do  i = 1, n
          pi(indx(i)) = ifix(float(i-1) / float(n) * 10.0)
        end do

        return
      end



      subroutine rdca(ioinp,brkname,x,y,z,maxres,nres,resnam,ierr)
        implicit none
        integer maxres,nres,ierr,ioinp
        real x(maxres),y(maxres),z(maxres)
        character resnam(maxres)*(*), resnum*5
        character card*60, atmnam*4, atnumb*5, brkname*(*)
        character test1*6,test2*4

        call openf(ioinp, brkname, 'old')
        nres = 0
5       read(ioinp, 10, end=20, err=300)card
10        format(a60)
          test1 = card(1:6)
          test2 = card(13:16)
          if ((test1.eq.'ATOM  ') .and. (test2.eq.' CA ')) then
            nres = nres + 1
            if (nres .gt. maxres) then
              ierr = 3
              write(*,*) 'Error[rdca]: too many residues'
              return
            end if
            read(card, 15) atnumb, atmnam, resnam(nres),
     *                     resnum, x(nres), y(nres), z(nres)
15          format(6x,a5,1x,a4,1x,a3,2x,a5,3x,3f8.3)
          end if
          go to 5
20      continue
        close(ioinp)

        write(*,30)nres
30      format(' PDB file read in; Nos. of res: ', i5)
        ierr = 0
        return

300     ierr = 2
        return
      end


c --- LPI-Fortran numerical precision fix:
      real function acosfix(aangle)
        implicit none
        real aangle
        if(aangle .lt. -1.0) aangle = -1.0
        if(aangle .gt.  1.0) aangle =  1.0
        acosfix = acos(aangle)
        return
      end


      subroutine inertia(x,y,z,n,r)
        implicit none
        integer n,i,j
        real x(n), y(n), z(n), r(3,3)

        do  i = 1,3
          do  j = 1, 3
            r(i,j) = 0.0
          end do
        end do

        do  i = 1, n
          r(1,1) = r(1,1) + y(i)*y(i) + z(i)*z(i)
          r(2,2) = r(2,2) + x(i)*x(i) + z(i)*z(i)
          r(3,3) = r(3,3) + x(i)*x(i) + y(i)*y(i)
          r(1,2) = r(1,2) - x(i)*y(i)
          r(1,3) = r(1,3) - x(i)*z(i)
          r(2,3) = r(2,3) - y(i)*z(i)
        end do
        r(2,1) = r(1,2)
        r(3,1) = r(1,3)
        r(3,2) = r(2,3)

        return
      end



      subroutine moment2(x,y,z,n,r)
        implicit none
        integer n,i,j
        real x(n), y(n), z(n), r(3,3)

        do  i = 1,3
          do  j = 1, 3
            r(i,j) = 0.0
          end do
        end do

        do  i = 1, n
          r(1,1) = r(1,1) + x(i)*x(i)
          r(2,2) = r(2,2) + y(i)*y(i)
          r(3,3) = r(3,3) + z(i)*z(i)
          r(1,2) = r(1,2) + x(i)*y(i)
          r(1,3) = r(1,3) + x(i)*z(i)
          r(2,3) = r(2,3) + y(i)*z(i)
        end do
        r(2,1) = r(1,2)
        r(3,1) = r(1,3)
        r(3,2) = r(2,3)

        return
      end
