c The IUPAC convention for sidechain dihedral angles is adopted, with
c the following exception: The atom numbered as epsilon position 1 in a 
c Brookhaven atom file is always used as the fourth point in the definition 
c of chi_2 without any swaps with epsilon position 2. Similarly for chi3.
c This introduces a difference with the true IUPAC value of at most of a
c couple degrees. Atom labels of Thr and Val chi1, and of Leu and ILE 
c chi2 are checked (chirality/pro-chirality).
c
c Note that dihedral angles chi2 of the DFY residues and chi_3 of E have
c almost perfect C2 symmetry and therefore the periodicity of chi_2/3 
c is really 180 degrees, not 360 degrees. However, the angle returned is 
c still in the -180 to 180 range to allow easier derivation and use of pdfs 
c in MODELLER where the angle can actually go from -180 to 180 to allow
c for free rotation of these sidechains. This is a change from the earlier
c definition which used the periodicity of 180 degrees.
c 
c The sidechain and mainchain dihedral angle classes are defined in 
c resdih.lib file. They were obtained by least-squares fitting the
c 5 degree distributions in ALBASE_1.list (structures with better or 
c equal to 2A resolution only). The classes are numbered 1, 2 and 3,
c in increasing order of frequency in the database.

cf ----------------------------------------------------------------------------
cf
cf    DIHEDS() returns dihedral angles of all types for all residues in 
cf    the protein. If DEGR=.T., they are returned in degrees, otherwise
cf    in radians. If CHKLBL=.T., the check of Thr and Ile chirality and
cf    some other checks are done. If ITYP=1, the chi2 dihedrals of Asp, 
cf    Tyr, Phe, and chi3 of Glu are calculated in the 0 .. 180 degree 
cf    range, taking into account the C2 symmetry of these dihedrals.
cf    Only the basic input data (coordinates, ...) are needed for this 
cf    routine.
cf
cf    subroutine diheds(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,natm,
cf   &           ityp,degr,chklbl,maxres,dih)
cf
cf ----------------------------------------------------------------------------

      subroutine diheds(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,natm,
     &           ityp,degr,chklbl,ndiht,maxres,dih)
      implicit none
#include "dihlib.cst"
      integer nres, iatmr1(nres), natm, ityp, idiht
      integer iresatm(natm), ndiht, maxres
      real dih(maxres,ndiht)
      real x(natm), y(natm), z(natm)
      character resnam(nres)*(*), atmnam(natm)*(*)
      logical degr,chklbl

        do  idiht = 1, ndiht
          call dihedn(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,
     &         natm,idiht,ityp,degr,chklbl,dih(1,idiht))
        end do
        call dihdisn(x,y,z,iatmr1,atmnam,resnam,nres,natm,degr,
     &       dih(1,ichi1typ),dih(1,ichi2typ),dih(1,ichi3typ),
     &       dih(1,ichi4typ),dih(1,ichi5typ))

      return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHEDN() returns dihedral angles of type IDIHT for all residues in 
cf    the protein. If DEGR=.T., they are returned in degrees, otherwise
cf    in radians. If CHKLBL=.T., the check of Thr and Ile chirality and
cf    some other checks are done. If ITYP=1, the chi2 dihedrals of Asp, 
cf    Tyr, Phe, and chi3 of Glu are calculated in the 0 .. 180 degree 
cf    range, taking into account the C2 symmetry of these dihedrals.
cf    Only the basic input data (coordinates, ...) are needed for this 
cf    routine. IDIHT specifies the entry in the dihedral angles library
cf    for each residue type (Normally: 1 = Phi, 2 = Psi, 3 = Omega, 
cf    4 = Chi1, 5 = Chi2, 6 = Chi3, 7 = Chi4).
cf
cf    subroutine dihedn(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,natm,
cf   -           idiht,ityp,degr,chklbl,dihn)
cf
cf ----------------------------------------------------------------------------

      subroutine dihedn(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,natm,
     &           idiht,ityp,degr,chklbl,dihn)
      implicit none
      integer ires, nres, iatmr1(nres), natm, ityp, idiht
      integer iresatm(natm), ierr
      real dihn(nres), dihedral3
      real x(natm), y(natm), z(natm)
      character resnam(nres)*(*), atmnam(natm)*(*)
      logical degr,chklbl

      do ires = 1, nres
        dihn(ires) = dihedral3(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,
     &                         natm,idiht,ityp,degr,chklbl,ires,ierr)
      end do

      return
      end


cf ----------------------------------------------------------------------------
cf
cf    DIHDISN() returns the chi1-5 dihedral angles for all disulfide
cf    bonded Cys residues in a molecule. If DEGR=.T., the angles are 
cf    returned in degrees, otherwise in radians. This routine relies
cf    on the existence of chi1 angles in chi1() obtained from a 
cf    prior call to dihdisn().
cf
cf    This routine is inefficient but it works.
cf
cf    subroutine dihdisn(x,y,z,iatmr1,atmnam,resnam,nres,natm,
cf   &           degr,chi1,chi2,chi3,chi4,chi5)
cf
cf ----------------------------------------------------------------------------

      subroutine dihdisn(x,y,z,iatmr1,atmnam,resnam,nres,natm,
     &           degr,chi1,chi2,chi3,chi4,chi5)
      implicit none
c --- maximal number of CYS residues in a protein
      integer maxss 
      parameter (maxss = 100)
#include "reslib.cst"
#include "dihlib.cst"
#include "reslib.cmn"
      integer nres,iatmr1(nres),natm,i,j,ichr2int,isgr(maxss)
      integer ierr1, ierr2, ierr3, iss(2,maxss),isg(maxss), nss, nsg
      integer indxatm
      real chi1(nres), chi2(nres), chi3(nres), chi4(nres), chi5(nres)
      real x(natm), y(natm), z(natm), dist1
      character resnam(nres)*(*), atmnam(natm)*(*)
      logical degr

c --- find the SG atom and residue indices:
      nsg = 0
      do  i = 1, nres
        if (ichr2int(resnam(i)).eq.icystyp) then
          j = indxatm('SG',i,atmnam,iatmr1,nres,natm)
          if (j .gt. 0) then
            nsg = nsg + 1
            if (nsg .gt. maxss) then
              write(*,'(a)') 'dihdisn_E> increase MAXSS'
              stop
            end if
            isg(nsg)  = j
            isgr(nsg) = i
          end if
        end if
      end do

c --- find the residue index pairs that are in a disulfide bond:
      nss = 0
      do  i = 1, nsg-1
        do 10  j = i+1, nsg
         if(dist1(x(isg(i)),y(isg(i)),z(isg(i)),
     &            x(isg(j)),y(isg(j)),z(isg(j))) .lt. dihdst) then
            nss = nss + 1
            iss(1,nss) = isgr(i)
            iss(2,nss) = isgr(j)
            go to 10
          end if
10      continue
      end do

      do i = 1, nss
        call dihdis(x,y,z,iatmr1,atmnam,nres,natm,degr,iss(1,i),
     &              iss(2,i),ierr1,ierr2,ierr3,chi2(iss(1,i)),
     &              chi3(iss(1,i)),chi4(iss(1,i)))

        chi5(iss(1,i)) = chi1(iss(2,i))
        chi2(iss(2,i)) = chi4(iss(1,i))
        chi3(iss(2,i)) = chi3(iss(1,i))
        chi4(iss(2,i)) = chi2(iss(1,i))
        chi5(iss(2,i)) = chi1(iss(1,i))

        if (max(ierr1,ierr2,ierr3) .gt. 0)
     &    write(*,'(a,2i5,5x,3i4)') 
     &    'dihdisn_W> disulfide angle chi2/3/4 non-existent:',
     &    iss(1,i),iss(2,i),ierr1,ierr2,ierr3
      end do

      return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHDIH() calculates the disulfide dihedral angles (chi2, DIH1;
cf    chi3, DIH2; chi4; dih3) for the C-C bridge IRES1--IRES2. 
cf    If DEGR=.T., dihedral angles are returned in degrees. 
cf    If any error is encountered, corresponding IERR? is > 0. 
cf
cf    subroutine dihdis(x,y,z,iatmr1,atmnam,nres,natm,degr,ires1,
cf   &                  ires2,ierr1,ierr2,ierr3,dih1,dih2,dih3)
cf
cf ----------------------------------------------------------------------------

      subroutine dihdis(x,y,z,iatmr1,atmnam,nres,natm,degr,ires1,
     &                  ires2,ierr1,ierr2,ierr3,dih1,dih2,dih3)
      implicit none
#include "numbers.cst"
      integer ires1, nres, iatmr1(nres), natm
      integer ierr1,indxatm,ierr2,ierr3,ica2
      integer ica1,icb1,icb2,isg1,isg2,ires2
      real dih1, dih2, dih3, degrees, x(natm), y(natm), z(natm)
      real xd(4), yd(4), zd(4), dihedral2
      character atmnam(natm)*(*)
      logical degr

      ica1 = indxatm('CA',ires1,atmnam,iatmr1,nres,natm)
      icb1 = indxatm('CB',ires1,atmnam,iatmr1,nres,natm)
      isg1 = indxatm('SG',ires1,atmnam,iatmr1,nres,natm)
      isg2 = indxatm('SG',ires2,atmnam,iatmr1,nres,natm)
      icb2 = indxatm('CB',ires2,atmnam,iatmr1,nres,natm)
      ica2 = indxatm('CA',ires2,atmnam,iatmr1,nres,natm)

      if (min(ica1,icb1,isg1,isg2).gt.0) then
        call gxyz(x,y,z,natm,ica1,icb1,isg1,isg2,xd,yd,zd)
        dih1 = dihedral2(xd,yd,zd,ierr1)
        if (degr.and.ierr1.eq.0) dih1 = degrees(dih1)
      else
        ierr1 = 10
        dih1 = nundf
      end if

      if (min(icb1,isg1,isg2,icb2).gt.0) then
        call gxyz(x,y,z,natm,icb1,isg1,isg2,icb2,xd,yd,zd)
        dih2 = dihedral2(xd,yd,zd,ierr2)
        if (degr.and.ierr2.eq.0) dih2 = degrees(dih2)
      else
        ierr2 = 10
        dih2 = nundf
      end if

      if (min(isg1,isg2,icb2,ica2).gt.0) then
        call gxyz(x,y,z,natm,isg1,isg2,icb2,ica2,xd,yd,zd)
        dih3 = dihedral2(xd,yd,zd,ierr3)
        if (degr.and.ierr3.eq.0) dih3 = degrees(dih3)
      else
        ierr3 = 10
        dih3 = nundf
      end if

      return
      end


cf ----------------------------------------------------------------------------
cf
cf    GXYZ() copies X(ICA1), Y(ICA1) ... to XD(1), YD(1), ...
cf
cf    subroutine gxyz(x,y,z,natm,ica1,icb1,isg1,isg2,xd,yd,zd)
cf
cf ----------------------------------------------------------------------------

      subroutine gxyz(x,y,z,natm,ica1,icb1,isg1,isg2,xd,yd,zd)
        implicit none
        integer natm,ica1,icb1,isg1,isg2
        real xd(4), yd(4), zd(4), x(natm), y(natm), z(natm)

        xd(1) = x(ica1)
        yd(1) = y(ica1)
        zd(1) = z(ica1)

        xd(2) = x(icb1)
        yd(2) = y(icb1)
        zd(2) = z(icb1)

        xd(3) = x(isg1)
        yd(3) = y(isg1)
        zd(3) = z(isg1)

        xd(4) = x(isg2)
        yd(4) = y(isg2)
        zd(4) = z(isg2)

        return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHEDRAL3() returns a dihedral angle of type IDIHT of residue IRES. 
cf    IERR<>0 if something is wrong. Other arguments the same as for 
cf    DIHEDN(). Only the basic input data (coordinates, ...) are needed 
cf    for this routine.
cf
cf    real function dihedral3(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,
cf   -                        natm,idiht,ityp,degr,chklbl,ires,ierr)
cf
cf ----------------------------------------------------------------------------

      real function dihedral3(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,
     &                        natm,idiht,ityp,degr,chklbl,ires,ierr)
      implicit none
#include "numbers.cst"
#include "reslib.cst"
#include "dihlib.cst"
#include "dihlib.cmn"
      integer ires, nres, ind(4), iatmr1(nres), natm, ityp, idiht, irest
      integer iresatm(natm),ierr,istr2int
      real dih, degrees, dihunq, dihchk
      real x(natm), y(natm), z(natm)
      character resnam(nres)*(*), atmnam(natm)*(*)
      logical degr, chklbl

      ierr = 0
      irest = istr2int(resnam(ires))

      dih = nundf

      if (irest.gt.0) then
       if (ndihlib(irest).ge.idiht) then

c ----- get the four atom indices and calculate the dihedral angle
        call dihedral5(x,y,z,iresatm,iatmr1,atmnam,nres,
     &                 natm,idiht,ires,irest,dih,ind,ierr)

        if (ierr .eq. 0) then
c ------- do I have to check the PDB atom labelling of some residues for
c         some dihedral angles?
          if (chklbl) dih=dihchk(x,y,z,iresatm,iatmr1,atmnam,resnam,
     &                           nres,natm,idiht,ires,dih,ierr)

c ------- SHOULD NOT BE USED ANYMORE:
c ------- do I have to take into account the C2 symmetry of some dihedral 
c         angle types:
          if (ityp .eq. 1) dih = dihunq(dih, resnam(ires), idiht)

c ------- do I have to convert the dihedral angle to degrees:
          if (degr) dih = degrees(dih)
        end if
       end if

      else
       ierr = 7
      end if

      dihedral3 = dih

c --- write out the error/warning messages if ierr<>0
      call errdih(ierr,ires,resnam,nres,idiht)
 
      return
      end





cf ----------------------------------------------------------------------------
cf
cf    DIHUNQN() routine makes the C2 dihedrals of type IDIHT unique 
cf    for all residues in a protein.
cf
cf    subroutine dihunqn(dih, resnam, nres, idiht)
cf
cf ----------------------------------------------------------------------------

      subroutine dihunqn(dih, resnam, nres, idiht)
        implicit none
        integer ires, nres, idiht
        real dih(nres), dihunq
        character resnam(nres)*(*)

        do  ires = 1, nres
          dih(ires) = dihunq(dih(ires), resnam(ires), idiht)
        end do

        return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHEDRAL5() returns a dihedral angle of type IDIHT, for residue IRES.
cf    Atom indices of the atoms defining the dihedral angle are returned in
cf    IND(4). The result is in radians.
cf
cf    subroutine dihedral5(x,y,z,iresatm,iatmr1,atmnam,nres,
cf   -                     natm,idiht,ires,irest,dih,ind,ierr)
cf
cf ----------------------------------------------------------------------------

      subroutine dihedral5(x,y,z,iresatm,iatmr1,atmnam,nres,
     &                     natm,idiht,ires,irest,dih,ind,ierr)
      implicit none
#include "numbers.cst"
#include "reslib.cst"
#include "dihlib.cst"
#include "dihlib.cmn"
      integer ires, nres, ind(4), iatmr1(nres), natm, idiht, irest
      integer iresatm(natm),ierr
      real dih
      real x(natm), y(natm), z(natm)
      character atmnam(natm)*(*)

      call dihedral6(x,y,z,iresatm,iatmr1,atmnam,nres,
     &               natm,ires,dihlib(1,idiht,irest),
     &               dih,ind,ierr)

      return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHEDRAL6() returns a dihedral angle in residue IRES, defined 
cf    by the atom types DIHLIB(4)*(5) (prefix-CHARMM atom names are 
cf    allowed referring to preceding and subsequent residue). The 
cf    result is in radians. The atom indices are returned in IND(4).
cf
cf    subroutine dihedral6(x,y,z,iresatm,iatmr1,atmnam,nres,
cf   -                     natm,ires,dihlib,dih,ind,ierr)
cf
cf ----------------------------------------------------------------------------

      subroutine dihedral6(x,y,z,iresatm,iatmr1,atmnam,nres,
     &                     natm,ires,dihlib,dih,ind,ierr)
        implicit none
#include "numbers.cst"
        integer ires, nres, ind(4), iatmr1(nres), natm
        integer iresatm(natm),ierr
        real dih, dihedral7
        real x(natm), y(natm), z(natm)
        character atmnam(natm)*(*), dihlib(4)*(*)
        logical present

c ----- find the four atom indices spanning the current dihedral angle:
        call fndatms(atmnam,iresatm,iatmr1,natm,nres,
     &               dihlib,4,ind,ires,1,present,.true.)

        if (present) then
c ------- calculate the dihedral angle for these four atoms:
          dih = dihedral7(x,y,z,natm,ind,4,ierr)
        else
c ------- some atoms are missing:
          ierr = 6
          dih = nundf
        end if

        return
      end


cf ----------------------------------------------------------------------------
cf
cf    DIHCHK() returns a corrected value of a chi1 (VAL/THR) or chi2 (LEU)
cf    dihedral angle value (input & output in radians), if the PDB 
cf    labelling would result in a wrong chirality. This function is a 
cf    very hardwired solution to the bad labelling problem. The routine
cf    should work with both the PDB and the CHARMM IUPAC atom names
cf    (ILE, LEU atoms are labelled slightly differently).
cf
cf    This routine compares fine with the SSTRUC results. 
cf
cf    real function dihchk(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,
cf   -                     natm,idiht,ires,dih,ierr)
cf
cf ----------------------------------------------------------------------------

      real function dihchk(x,y,z,iresatm,iatmr1,atmnam,resnam,nres,
     &                     natm,idiht,ires,dih,ierr)
        implicit none
#include "numbers.cst"
#include "reslib.cst"
#include "dihlib.cst"
#include "dihlib.cmn"
        integer idiht, ierr, ires, natm, nres, iatmr1(nres)
        integer iresatm(natm), ind(4)
        real dih, dih2, x(natm), y(natm), z(natm)
        character resnam(nres)*(*), dihlb(4)*(4), atmnam(natm)*(*)
        logical clock180

        dihchk = dih
        ierr = 0

c ----- chi1 VAL CG1/CG2, THR OG1/CG2, ILE CG1/CG2 labelling:
        if (idiht .eq. ichi1typ) then

          if (index('VAL', resnam(ires)) .gt. 0) then
            dihlb(1) = 'N '
            dihlb(2) = 'CA'
            dihlb(3) = 'CB'
            dihlb(4) = 'CG2'
            call dihedral6(x,y,z,iresatm,iatmr1,atmnam,nres,
     &                     natm,ires,dihlb,dih2,ind,ierr)
            if (clock180(dih, dih2, pi, pi2)) then
              dihchk = dih2
              ierr = 8
            end if
          end if

          if (index('THR', resnam(ires)) .gt. 0) then
            dihlb(1) = 'N '
            dihlb(2) = 'CA'
            dihlb(3) = 'CB'
            dihlb(4) = 'CG2'
            call dihedral6(x,y,z,iresatm,iatmr1,atmnam,nres,
     &                     natm,ires,dihlb,dih2,ind,ierr)
            if (.not. clock180(dih, dih2, pi, pi2)) then
              dihchk = dih2
              ierr = 8
            end if
          end if

          if (index('ILE', resnam(ires)) .gt. 0) then
            dihlb(1) = 'N '
            dihlb(2) = 'CA'
            dihlb(3) = 'CB'
            dihlb(4) = 'CG2'
            call dihedral6(x,y,z,iresatm,iatmr1,atmnam,nres,
     &                     natm,ires,dihlb,dih2,ind,ierr)
            if (.not. clock180(dih, dih2, pi, pi2)) then
              dihchk = dih2
              ierr = 8
            end if
          end if

        end if


c ----- chi2 LEU [CD1/CD2] labelling (should work with both the PDB and
c       CHARMM conventions):
        if (idiht .eq. ichi2typ) then

          if (index('LEU', resnam(ires)) .gt. 0) then
c --------- To make it work with both PDB and CHARMM labelling, calculate
c           dih for the CD2 possibility again (just in case you did not 
c           convert the input PDB file to CHARMM atom names); Note that
c           the resdih.lib expects the CHARMM atom names and consequently
c           specifies the chi_2 with the CD2 atom name.
            dihlb(1) = 'CA'
            dihlb(2) = 'CB'
            dihlb(3) = 'CG'
            dihlb(4) = 'CD2'
            call dihedral6(x,y,z,iresatm,iatmr1,atmnam,nres,
     &                     natm,ires,dihlb,dih,ind,ierr)

            dihlb(1) = 'CA'
            dihlb(2) = 'CB'
            dihlb(3) = 'CG'
            dihlb(4) = 'CD1'
            call dihedral6(x,y,z,iresatm,iatmr1,atmnam,nres,
     &                     natm,ires,dihlb,dih2,ind,ierr)
            if (clock180(dih, dih2, pi, pi2)) then
              dihchk = dih2
              ierr = 8
            end if
          end if

        end if

        return
      end




cf ----------------------------------------------------------------------------
cf
cf    CLOCK180() returns .T. if more than PI angular units are needed 
cf    to rotate ANG1 to ANG2 in a positive direction. PI2 has to be 
cf    2*PI, either in radians or degrees.
cf
cf    logical function clock180(ang1, ang2, pi, pi2)
cf
cf ----------------------------------------------------------------------------


      logical function clock180(ang1, ang2, pi, pi2)
        implicit none
        real ang1, ang2, pi, pi2, ang

c ----- put ang2 on such a period that it is numerically larger than ang1
c       (do not erase ang2!)
        if (ang2 .lt. ang1) then
          ang = ang2 + pi2
        else
          ang = ang2
        end if

c ----- test
        if ((ang-ang1) .gt. pi) then
          clock180 = .true.
        else
          clock180 = .false.
        end if

        return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHUNQ() makes the dihedral angles involving the sp2 center
cf    at position gamma (ASP, PHE, TYR) or position delta (GLU) 
cf    unique: C2 symmetry is taken into account by translating the 
cf    angle for Pi radians, if necessary, so that the result is 
cf    always in the range from 0 to Pi radians. Assuming that the
cf    argument is always in the range from -Pi to Pi radians.
cf
cf    real function dihunq(dih, resnam, idiht)
cf
cf ----------------------------------------------------------------------------

      real function dihunq(dih, resnam, idiht)
        implicit none
#include "numbers.cst"
#include "reslib.cst"
#include "dihlib.cst"
#include "dihlib.cmn"
        integer idiht
        real dih
        character resnam*(*)

        dihunq = dih

c ----- for chi2:       
        if (idiht.eq.ichi2typ) then
          if (index('ASP TYR PHE', resnam) .gt. 0) then
            if (dih .lt. 0.0) dihunq = dih + pi
          end if
        end if

c ----- for chi3:       
        if (idiht.eq.ichi3typ) then
          if (index('GLU', resnam) .gt. 0) then
            if (dih .lt. 0.0) dihunq = dih + pi
          end if
        end if

        return
      end

     

cf ----------------------------------------------------------------------------
cf
cf    DIHEDRAL7() uses coordinates of atoms with indices IND(NIND=4)
cf    and calculates the dihedral angle in radians. If something is
cf    wrong, IERR<>0.
cf
cf    real function dihedral7(x,y,z,natm,ind,nind,ierr)
cf
cf ----------------------------------------------------------------------------

      real function dihedral7(x,y,z,natm,ind,nind,ierr)
      implicit none
      integer nind, natm, ind(nind), i, ierr
      real x(natm), y(natm), z(natm), xd(4), yd(4), zd(4), dihedral2

      do  i = 1, 4
        xd(i) = x(ind(i))
        yd(i) = y(ind(i))
        zd(i) = z(ind(i))
      end do

      dihedral7 = dihedral2(xd,yd,zd,ierr)

      return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHEDRAL() uses the coordinates of atoms with indices IND(NIND=4)
cf    and returns the dihedral angle in radians.
cf
cf    real function dihedral(x,y,z,natm,ind,nind)
cf
cf ----------------------------------------------------------------------------

      real function dihedral(x,y,z,natm,ind,nind)
      implicit none
      integer nind, natm, ind(nind), i, ierr
      real x(natm), y(natm), z(natm), xd(4), yd(4), zd(4), dihedral2

      do  i = 1, 4
        xd(i) = x(ind(i))
        yd(i) = y(ind(i))
        zd(i) = z(ind(i))
      end do

      dihedral = dihedral2(xd,yd,zd,ierr)

      return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHEDRAL2() uses the coordinates of atoms X(4),Y(4),Z(4) and returns
cf    the dihedral angle in radians. IERR<>0 if something wrong.
cf
cf    real function dihedral2(x,y,z,ierr)
cf
cf ----------------------------------------------------------------------------


      real function dihedral2(x,y,z,ierr)
      implicit none
#include "numbers.cst"
      integer ierr
      real x(4), y(4), z(4), vsize2, scalr2
      real v1(3), v2(3), v(3), sgn, v12, v1siz, v2siz, v12siz, v12norm
      real rij(3), rkj(3), rkl(3)

c --- get the three vectors defining the dihedral angle:
      rij(1) = x(1) - x(2)
      rij(2) = y(1) - y(2)
      rij(3) = z(1) - z(2)

      rkj(1) = x(3) - x(2)
      rkj(2) = y(3) - y(2)
      rkj(3) = z(3) - z(2)

      rkl(1) = x(3) - x(4)
      rkl(2) = y(3) - y(4)
      rkl(3) = z(3) - z(4)

c --- get the two vector products you need:
      call vectr2(rij, rkj, v1)
      call vectr2(rkj, rkl, v2)

c --- get the sign:
      call vectr2(v1, v2, v)
      sgn = scalr2(rkj, v)

c --- calculate the cosine of the dihedral angle:
      v12 = scalr2(v1, v2)
      v1siz = vsize2(v1)
      v2siz = vsize2(v2)
      v12siz = v1siz*v2siz
c --- check the denominator for not being 0.0
      if (v12siz .lt. tenm4) then
        ierr = 1
c        write(*,'(a)') 'dihedra_E> check the arguments'
       dihedral2 = nundf
       return 
      end if

      v12norm = v12 / v12siz
      v12norm = min(1.0, max(-1.0, v12norm))

      ierr = 0
      dihedral2 = sign(1.0, sgn) * acos(v12norm)

c        write(*,*) 'DIH: ', dih
c        write(*,*) x(ind(1)),x(ind(2)),x(ind(3)),x(ind(4))
c        write(*,*) y(ind(1)),y(ind(2)),y(ind(3)),y(ind(4))
c        write(*,*) z(ind(1)),z(ind(2)),z(ind(3)),z(ind(4))
c        write(*,*) 'Bond 1: ', rij(1),rij(2),rij(3)
c        write(*,*) 'Bond 2: ', rkj(1),rkj(2),rkj(3)
c        write(*,*) 'Bond 3: ', rkl(1),rkl(2),rkl(3)
c        write(*,*) 'V1    : ', v1
c        write(*,*) 'V2    : ', v2
c        write(*,*) 'V1xV2 : ', v
c        write(*,*) 'V1*V2 : ', v12
c        write(*,*) 'V1,V2 : ', v1siz,v2siz
c        write(*,*) 'V12NORM, acos(V12NORM): ', v12norm, acos(v12norm)

c     dihedral2 = raddeg * sign(1.0, sgn) * acos(v12norm)

      return
      end


cf ----------------------------------------------------------------------------
cf
cf    DIHEDRAL4() uses the coordinates of atoms X(4),Y(4),Z(4) and returns
cf    the dihedral angle in radians. IERR<>0 if something wrong. Error 
cf    checking assumes that the atoms form three consecutive chemical
cf    bonds. This is a slower, more specialized, and more error robust
cf    version of DIHEDRAL2().
cf
cf    real function dihedral4(x,y,z,ierr)
cf
cf ----------------------------------------------------------------------------

      real function dihedral4(x,y,z,ierr)
      implicit none
#include "numbers.cst"
#include "dihlib.cst"
      real x(4), y(4), z(4), dist1, dihedral2
      integer ierr
      logical undefd

      dihedral4 = nundf
      ierr = 0

c --- if coordinates for any of the atoms are missing then return ierr=2

      if(undefd(x(1)).or.undefd(x(2)).or.undefd(x(3)).or.
     &   undefd(x(4)))then
        ierr = 2
        return
      end if

c --- Check the bond lengths to avoid chain breaks or missing bonds
      if(dist1(x(1),y(1),z(1),x(2),y(2),z(2)).gt.dihdst)then
        ierr = 3
        return
      end if
      if(dist1(x(3),y(3),z(3),x(2),y(2),z(2)).gt.dihdst)then
        ierr = 4
        return
      end if
      if(dist1(x(3),y(3),z(3),x(4),y(4),z(4)).gt.dihdst)then
        ierr = 5
        return
      end if

      dihedral4 = dihedral2(x,y,z,ierr)

      return
      end




cf ----------------------------------------------------------------------------
cf
cf    ERRDIH() reports the appropriate dihedral angle calculation error
cf    message given the error code IERR.
cf
cf    subroutine errdih(ierr,ires,resnam,nres,idiht)
cf
cf ----------------------------------------------------------------------------

      subroutine errdih(ierr,ires,resnam,nres,idiht)
        implicit none
#include "dihlib.cst"
#include "reslib.cst"
#include "reslib.cmn"
        integer nres, ierr, ires, idiht, istr2int
        character resnam(nres)*(*)
        logical iowr

        if (ierr .gt. 0) then

          go to (1,2,3,3,3,6,7,8) ierr
1         if (iowr(4)) write(*,'(a,2i5,1x,a)') 
     &    'errdih__E> SIZE1*SIZE2 = 0',idiht,ires,resnam(ires)
          go to 100

2         if (iowr(4)) write(*,'(a,2i5,1x,a)')
     &    'errdih__E> At least one atom has undefined XYZ: ',
     &    idiht,ires,resnam(ires)
          go to 100

3         if (iowr(4)) write(*,'(a,3i5,1x,a)') 
     &    'errdih__E> Bond i too long: ',ierr-2,idiht,ires,resnam(ires)
          go to 100

6         continue
c ------- not an error for the chi2 and chi3 of Cys; Phi of 1, and Psi and
c         omega of NRES residue:
          if (iowr(3)) then
            if (.not. (((istr2int(resnam(ires)).eq.icystyp).and.
     &         (idiht.ge.ichi2typ)) .or. 
     &         ((idiht.eq.iphityp).and.(ires.eq.1)) .or.
     &         ((idiht.eq.ipsityp).and.(ires.eq.nres)) .or.
     &         ((idiht.eq.iomgtyp).and.(ires.eq.nres))))
     &      write(*,'(a,2i5,1x,a)')
     &      'errdih__W> Missing atoms: ', idiht, ires, resnam(ires)
          end if
          go to 100

7         if (iowr(4)) write(*,'(a,2i5,1x,a)') 
     &    'errdih__E> Residue type not recognized: ',
     &    idiht,ires,resnam(ires)
          go to 100

8         if (iowr(3)) write(*,'(a,2i5,1x,a)') 
     &    'errdih__W> Atoms are labelled incorrectly; swapped:',
     &    idiht,ires,resnam(ires)
          go to 100

100       continue

        end if
 
        return
      end



cf ----------------------------------------------------------------------------
cf
cf    DIHCLSN() calculates the dihedral angle class of each residue 
cf    for dihedral angle type IDIHT. Dihedral angle classes are defined 
cf    in the DIHLIB library file, in degrees if DEGRLIB=.T.
cf
cf    subroutine dihclsn(nres,dih,irestyp,idiht,idih,degr,degrlib)
cf
cf ----------------------------------------------------------------------------

      subroutine dihclsn(nres,dih,irestyp,idiht,idih,degr,degrlib)
        implicit none
        integer i, nres, idih(nres), idihcls, idiht, irestyp(nres)
        real dih(nres)
        logical degr,degrlib
         
        do  i = 1, nres
          idih(i) = idihcls(dih(i), irestyp(i), idiht, degr, degrlib)
        end do

        return
      end



cf ----------------------------------------------------------------------------
cf
cf    IDIHCLS() returns a dihedral angle class for angle DIH of dihedral
cf    angle type IDIHT for residue type RESNAM*(3). If DEGR = .T., the 
cf    argument DIH is in degrees. The dihedral angle class is found by 
cf    looking at the angle ranges in the library of the residue dihedrals. 
cf    The values in the library are in degrees if DEGRLIB=.T. If residue
cf    type is undefined, or the angle is not in any of the angle ranges,
cf    the undefined class equal to MDIHTYP is returned. MDIHTYP must be
cf    larger than any of the defined angle classes (it is 4 at the moment).
cf
cf    integer function idihcls(dih, irestyp, idiht, degr, degrlib)
cf
cf ----------------------------------------------------------------------------

      integer function idihcls(dih, irestyp, idiht, degr, degrlib)
        implicit none
#include "numbers.cst"
#include "reslib.cst"
#include "dihlib.cst"
#include "dihlib.cmn"
        integer idiht, ic, irestyp
        real dih,dihd,degrees,angdeg,period,angrad,radians
        logical degr,degrlib

c ----- convert the argument and periodicity into the same units as the library
        if (degrlib) then
          period = pi2degr
          if (degr) then
            dihd = angdeg(dih)
          else
            dihd = degrees(dih)
          end if
        else
          period = pi2
          if (degr) then
            dihd = radians(dih)
          else
            dihd = angrad(dih)
          end if
        end if

        if (irestyp .gt. 0) then
          do  ic = 1, ndihopt(idiht,irestyp)
c --------- shift the allowed interval up and down for 2 Pi:
            if (((dihd       .ge.dihcls(1,ic,idiht,irestyp)) .and.  
     &           (dihd       .le.dihcls(2,ic,idiht,irestyp))) .or.
     &          ((dihd+period.ge.dihcls(1,ic,idiht,irestyp)) .and.  
     &           (dihd+period.le.dihcls(2,ic,idiht,irestyp))) .or.
     &          ((dihd-period.ge.dihcls(1,ic,idiht,irestyp)) .and.  
     &           (dihd-period.le.dihcls(2,ic,idiht,irestyp)))) then
              idihcls = ic
              go to 100
            end if
          end do
          idihcls = ndihc + 1
        else
          idihcls = ndihc + 1
        end if

100     continue

        return
      end



cf ----------------------------------------------------------------------------
cf
cf    SC12DIHBIN() returns in ICHI(NRES) the sidechain conformation classes
cf    for all residues given their chi1 and chi2 dihedral angles and their
cf    residue types. The value is currently in the range from 1 to 16 and
cf    is calculated from the separate classes for chi1 and chi2 angles.
cf    It assumes that all defined angle classes chi1 and chi2 are either
cf    1, 2, or 3, and that the undefined class is 4.
cf
cf    subroutine sc12dihbin(nres,chi1,chi2,irestyp,ichi,degr,degrlib)
cf
cf ----------------------------------------------------------------------------

      subroutine sc12dihbin(nres,chi1,chi2,irestyp,ichi,degr,degrlib)
        implicit none
#include "reslib.cst"
#include "dihlib.cst"
#include "dihlib.cmn"
        integer nres,i,i1,i2,ichi(nres),idihcls,irestyp(nres)
        real chi1(nres), chi2(nres)
        logical degr,degrlib
        do  i = 1, nres
          i1 = idihcls(chi1(i),irestyp(i),ichi1typ,degr,degrlib)
          i2 = idihcls(chi2(i),irestyp(i),ichi2typ,degr,degrlib)
c ------- just in case limit idihcls to 4 (sdch1.lib library of only 16 
c         chi12 classes is used):
          ichi(i) = min(i1,4) + (min(i2,4)-1)*4
        end do
        return
      end


cf ----------------------------------------------------------------------------
cf
cf    MCCHNFBIN() returns the mainchain conformation classes in IMNCH(NRES) 
cf    for all residues, given the Phi and Psi dihedral angles and secondary 
cf    structure assignments.
cf
cf    subroutine mccnfbin(nres,phi,psi,ssec,imnch)
cf
cf ----------------------------------------------------------------------------

      subroutine mccnfbin(nres,phi,psi,ssec,imnch)
        implicit none
        integer nres, i, imnch(nres), iclsmcbin
        real phi(nres), psi(nres) 
        character ssec(nres)*1
        do 10  i = 1, nres
          imnch(i) = iclsmcbin(phi(i), psi(i), ssec(i))
10      continue
        return
      end

c --- Wilmot class:
      subroutine mccnfwbin(nres,phi,psi,imnch,irestyp)
        implicit none
        integer nres,i,imnch(nres), iwclsmcbin, irestyp(nres)
        real phi(nres), psi(nres) 
        do 10  i = 1, nres
          imnch(i) = iwclsmcbin(phi(i), psi(i), irestyp(i))
10      continue
        return
      end

cf ----------------------------------------------------------------------------
cf
cf    ICLSMCBIN() returns the Alpha,Beta,Other,+Phi integer index for 
cf    the mainchain conformation.
cf
cf    integer function iclsmcbin(phi,psi,ssec)
cf
cf ----------------------------------------------------------------------------

      integer function iclsmcbin(phi,psi,ssec)
        implicit none
#include "numbers.cst"
        real phi,psi
        character ssec*1
        logical defd

        if((phi .gt. 0) .and. defd(phi))then
                                                  iclsmcbin=4
        else
          if((ssec.eq.'H').or.(ssec.eq.'G').or.
     &       (ssec.eq.'I'))                       iclsmcbin=1
          if((ssec.eq.'E').or.(ssec.eq.'B').or.
     &       (ssec.eq.' '))                       iclsmcbin=2
          if((ssec.eq.'T').or.(ssec.eq.'S'))      iclsmcbin=3
        endif

        return
      end


cf ----------------------------------------------------------------------------
cf
cf    IWCLSMCBIN() returns the bin index for the mainchain conformation
cf    using the definitions based on the areas in the Ramchandran plot.
cf
cf    integer function iwclsmcbin(phia,psia,irtyp)
cf
cf ----------------------------------------------------------------------------

      integer function iwclsmcbin(phia,psia,irtyp)
        implicit none
#include "mdt_mdt.cst"
#include "mdt_mdt.cmn"
        integer indphi,indpsi,irtyp
        real phia,psia,angdeg,phi180,psi180
        logical defd

        psi180 = angdeg(psia)
        phi180 = angdeg(phia)

        if (defd(phi180) .and. defd(psi180) .and. irtyp.gt.0 .and.
     &      irtyp .le. 20) then
          indphi = int(phi180 / 10.0) + 19
          indpsi =-int(psi180 / 10.0) + 19
          if (indphi.eq.37) indphi = 36
          if (indpsi.eq.37) indpsi = 36
          iwclsmcbin = iwilmot(indphi,indpsi,irtyp)
        else
          iwclsmcbin = nmnch2 + 1
        end if 

        return
      end


      subroutine wrdih(ioout,iolog,fname,dihn,resnam,maxres,nres,ndiht)
        implicit none
        integer i,k,ioout,nres,maxres,ndiht,ierr,iolog
        real dihn(maxres,ndiht)
        character fname*(*), resnam(nres)*(*)
        logical cmpr

        call openf4(ioout,fname,'UNKNOWN','SEQUENTIAL','FORMATTED',3,
     &              .true.,ierr,cmpr,iolog)
        write(ioout,'(a,i4)') 'TOTAL RES: ', nres
        write(ioout,'(2a)') 
     -  '   #   RES    PHI    PSI  OMEGA  CHI_1  CHI_2  CHI_3  ',
     -  'CHI_4  CHI_5'
        do  i = 1, nres
          write(ioout, '(i4, 3x, a4, 999f7.1)') 
     -          i, resnam(i), (dihn(i,k),k=1,ndiht)
        end do
        close(ioout)

        return
      end
