import _modeller
import model
import profile
import util.modutil as modutil
import util.modlist as modlist
from modeller.util.modobject import modobject
from modeller.coordinates import coordinates
from modeller import sequence

__docformat__ = "epytext en"

class alignment(modobject):
    """Holds an alignment of protein sequences"""
    __modpt = None
    env = None

    def __init__(self, env, **vars):
        self.__modpt = _modeller.mod_alignment_new(self)
        self.env = env.copy()
        if len(vars) > 0:
            self.append(**vars)

    def __repr__(self):
        if len(self) == 0:
            return "Empty alignment"
        else:
            return "Alignment of " + ", ".join([repr(s) for s in self])

    def __str__(self):
        if len(self) == 0:
            return "<Empty alignment>"
        else:
            return "<Alignment of " + ", ".join([str(s) for s in self]) + ">"

    def __setstate__(self, d):
        self.__dict__.update(d)
        self.__modpt = _modeller.mod_alignment_new(self)

    def __del__(self):
        if self.__modpt:
            _modeller.mod_alignment_free(self.__modpt)

    def __get_modpt(self):
        return self.__modpt

    def __len__(self):
        return _modeller.mod_alignment_nseq_get(self.modpt)

    def clear(self):
        """Remove all sequences from the alignment"""
        return _modeller.mod_alignment_clear(self.modpt)

    def append(self, file, align_codes='all', atom_files=None, remove_gaps=True,
               alignment_format='PIR', rewind_file=False, close_file=True,
               io=None):
        """Add sequence(s) from an alignment file"""
        if io is None:
            io = self.env.io
        new_align_codes = align_codes
        new_atom_files = atom_files
        align_codes = [seq.code for seq in self]
        atom_files = [seq.atom_file for seq in self]
        if new_align_codes is not None:
            if not isinstance(new_align_codes, (tuple, list)):
                new_align_codes = [new_align_codes]
            align_codes += new_align_codes
        if new_atom_files is not None:
            if not isinstance(new_atom_files, (tuple, list)):
                new_atom_files = [new_atom_files]
            atom_files += new_atom_files
        func = _modeller.mod_alignment_read
        return func(self.modpt, io.modpt, self.env.libs.modpt, align_codes,
                    atom_files, file, remove_gaps, alignment_format,
                    rewind_file, close_file)

    def read(self, **vars):
        """Read sequence(s) from an alignment file"""
        self.clear()
        return self.append(**vars)

    def read_one(self, **vars):
        """Read sequences from file, one by one. Return False when no more
           can be read."""
        vars['align_codes'] = '@oNe@'
        vars['close_file'] = False
        return (self.read(**vars) == 0)

    def append_model(self, mdl, align_codes, atom_files=''):
        """Add a sequence from a model"""
        return _modeller.mod_alignment_append_model(self.modpt, mdl.modpt,
                                                    self.env.libs.modpt,
                                                    atom_files, align_codes)

    def append_sequence(self, sequence):
        """Add a new sequence, as a string of one-letter codes"""
        return _modeller.mod_alignment_append_sequence(self.modpt, sequence,
                                                       self.env.libs.modpt)

    def compare_with(self, aln):
        """Compare with another alignment"""
        return _modeller.mod_alignment_compare_with(aln=self.modpt,
                                                    aln2=aln.modpt)

    def compare_structures(self, compare_mode=3, fit=True, fit_atoms='CA',
                           matrix_file='family.mat', output='LONG',
                           asgl_output=False, refine_local=True,
                           rms_cutoffs=(3.5, 3.5, 60., 60., 15., 60., 60.,
                                        60., 60., 60., 60.),
                           varatom='CA', edat=None, io=None):
        """Compare 3D structures"""
        if io is None:
            io = self.env.io
        if edat is None:
            edat = self.env.edat
        func = _modeller.mod_compare_structures
        return func(self.modpt, edat.modpt, io.modpt, self.env.libs.modpt,
                    compare_mode, fit, fit_atoms, matrix_file, output,
                    asgl_output, refine_local, rms_cutoffs, varatom)

    def compare_sequences(self, mdl, matrix_file, variability_file,
                          max_gaps_match, rr_file='${LIB}/as1.sim.mat'):
        """Compare sequences in alignment"""
        return _modeller.mod_compare_sequences(self.modpt, mdl.modpt,
                                               self.env.libs.modpt, rr_file,
                                               matrix_file, variability_file,
                                               max_gaps_match)

    def segment_matching(self, file, root_name, file_ext, file_id, align_block,
                         segment_report, segment_cutoff, segment_shifts,
                         segment_growth_n, segment_growth_c, min_loop_length,
                         rr_file='${LIB}/as1.sim.mat'):
        """Enumerates alignments between two blocks of sequences.
           More precisely, it enumerates the alignments between the segments in
           the first block, as defined by C{align_block}, and the sequences in
           the second block. The segments can be moved to the left and right as
           well as lengthened and shortened, relative to the initial alignment.
        """
        return _modeller.mod_segment_matching(self.modpt, self.env.libs.modpt,
                                              rr_file, file, align_block,
                                              min_loop_length, segment_shifts,
                                              segment_growth_n,
                                              segment_growth_c,
                                              segment_cutoff, segment_report,
                                              root_name, file_id, file_ext)

    def edit(self, overhang, edit_align_codes, base_align_codes,
             min_base_entries, io=None):
        """Edit overhangs in alignment.
           @param edit_align_codes: specifies the alignment codes for the
                  alignment entries whose overhangs are to be cut; in
                  addition, C{all} or C{last} can be used.
           @param base_align_codes: specifies the alignment codes for the
                  alignment entries that are used to determine the extent of
                  the overhangs to be cut from the edited entries; in
                  addition, C{all} or C{rest} (relative to C{edit_align_codes})
                  can be used.
        """
        if io is None:
            io = self.env.io
        return _modeller.mod_alignment_edit(self.modpt, io.modpt,
                                            self.env.libs.modpt, overhang,
                                            edit_align_codes, base_align_codes,
                                            min_base_entries)

    def append_profile(self, prf):
        """Add sequences from a profile"""
        return _modeller.mod_profile_to_aln(prf=prf.modpt, aln=self.modpt,
                                            libs=self.env.libs.modpt)

    def to_profile(self):
        """Converts the alignment to profile format"""
        prf = profile.profile(self.env)
        _modeller.mod_profile_from_aln(aln=self.modpt, prf=prf.modpt,
                                       libs=self.env.libs.modpt)
        return prf

    def check(self, io=None):
        """Check alignment for modeling"""
        if io is None:
            io = self.env.io
        return _modeller.mod_alignment_check(aln=self.modpt, io=io.modpt,
                                             libs=self.env.libs.modpt)

    def describe(self, io=None):
        """Describe proteins"""
        if io is None:
            io = self.env.io
        return _modeller.mod_alignment_describe(aln=self.modpt, io=io.modpt,
                                                libs=self.env.libs.modpt)

    def consensus(self, align_block=0, gap_penalties_1d=(-900., -50.),
                  max_gap_length=999999, weigh_sequences=False,
                  input_weights_file=None, output_weights_file=None,
                  weights_type='SIMILAR', smooth_prof_weight=10):
        """Produce a consensus alignment"""
        (input_weights_file, read_weights) = self.__opt_file(input_weights_file)
        (output_weights_file,
         write_weights) = self.__opt_file(output_weights_file)
        func = _modeller.mod_alignment_consensus
        return func(self.modpt, self.env.libs.modpt, align_block,
                    gap_penalties_1d, max_gap_length, read_weights,
                    write_weights, weigh_sequences, input_weights_file,
                    output_weights_file, weights_type, smooth_prof_weight)

    def __opt_file(self, fname):
        """Processing for optional files for some routines"""
        if fname is None or fname == "":
            return ('', False)
        else:
            return (fname, True)

    def align(self, off_diagonal=100, max_gap_length=999999,
              local_alignment=False, matrix_offset=0.,
              gap_penalties_1d=(-900., -50.), n_subopt=1, subopt_offset=0.,
              weigh_sequences=False, smooth_prof_weight=10, align_what='BLOCK',
              weights_type='SIMILAR', input_weights_file=None,
              output_weights_file=None, rr_file='$(LIB)/as1.sim.mat',
              overhang=0, align_block=0):
        """Align two (blocks of) sequences"""
        (input_weights_file, read_weights) = self.__opt_file(input_weights_file)
        (output_weights_file,
         write_weights) = self.__opt_file(output_weights_file)
        func = _modeller.mod_align
        return func(self.modpt, self.env.libs.modpt, off_diagonal,
                    max_gap_length, local_alignment, matrix_offset,
                    gap_penalties_1d, read_weights, write_weights, n_subopt,
                    subopt_offset, weigh_sequences, smooth_prof_weight,
                    align_what, weights_type, input_weights_file,
                    output_weights_file, rr_file, overhang, align_block)

    def malign(self, rr_file='$(LIB)/as1.sim.mat', off_diagonal=100,
               local_alignment=False, matrix_offset=0., overhang=0,
               align_block=0, gap_penalties_1d=(-900., -50.)):
        """Align two or more sequences"""
        return _modeller.mod_malign(self.modpt, self.env.libs.modpt, rr_file,
                                    off_diagonal, local_alignment,
                                    matrix_offset, overhang, align_block,
                                    gap_penalties_1d)

    def align2d(self, overhang=0, align_block=0, rr_file='$(LIB)/as1.sim.mat',
                align_what='BLOCK', off_diagonal=100, max_gap_length=999999,
                local_alignment=False, matrix_offset=0.,
                gap_penalties_1d=(-900., -50.),
                gap_penalties_2d=(3.5, 3.5, 3.5, 0.2, 4.0, 6.5, 2.0, 0., 0.),
                surftyp=1, fit=True, fix_offsets=(0., -1., -2., -3., -4.),
                input_weights_file=None, output_weights_file=None, n_subopt=1,
                subopt_offset=0., input_profile_file=None,
                output_profile_file=None, weigh_sequences=False,
                smooth_prof_weight=10, weights_type='SIMILAR', io=None):
        """Align sequences with structures"""
        if io is None:
            io = self.env.io
        (input_weights_file, read_weights) = self.__opt_file(input_weights_file)
        (output_weights_file,
         write_weights) = self.__opt_file(output_weights_file)
        (input_profile_file, read_profile) = self.__opt_file(input_profile_file)
        (output_profile_file,
         write_profile) = self.__opt_file(output_profile_file)

        func = _modeller.mod_align2d
        return func(self.modpt, io.modpt, self.env.libs.modpt, overhang,
                    align_block, rr_file, align_what, off_diagonal,
                    max_gap_length, local_alignment, matrix_offset,
                    gap_penalties_1d, gap_penalties_2d, surftyp, fit,
                    fix_offsets, read_weights, write_weights,
                    input_weights_file, output_weights_file, n_subopt,
                    subopt_offset, read_profile, input_profile_file,
                    write_profile, output_profile_file, weigh_sequences,
                    smooth_prof_weight, weights_type)

    def align3d(self, off_diagonal=100, overhang=0, local_alignment=False,
                matrix_offset=0., gap_penalties_3d=(0.0, 1.75), fit=True,
                fit_atoms='CA', align3d_trf=False, output='LONG',
                align3d_repeat=False, io=None):
        """Align two structures"""
        if io is None:
            io = self.env.io
        func = _modeller.mod_align3d
        return func(self.modpt, io.modpt, self.env.libs.modpt, off_diagonal,
                    overhang, local_alignment, matrix_offset, gap_penalties_3d,
                    fit, fit_atoms, align3d_trf, output, align3d_repeat)

    def malign3d(self, off_diagonal=100, overhang=0, local_alignment=False,
                 matrix_offset=0., gap_penalties_3d=(0.0, 1.75), fit=True,
                 fit_atoms='CA', output='LONG', write_whole_pdb=True,
                 current_directory=True, write_fit=False,
                 edit_file_ext=('.pdb', '_fit.pdb'), io=None):
        """Align two or more structures"""
        if io is None:
            io = self.env.io
        return _modeller.mod_malign3d(self.modpt, io.modpt, self.env.libs.modpt,
                                      off_diagonal, overhang, local_alignment,
                                      matrix_offset, gap_penalties_3d, fit,
                                      fit_atoms, output, write_whole_pdb,
                                      current_directory, write_fit,
                                      edit_file_ext)

    def salign(self, residue_type2='REGULAR', no_ter=False, overhang=0,
               off_diagonal=100, matrix_offset=0.,
               gap_penalties_1d=(-900., -50.),
               gap_penalties_2d=(3.5, 3.5, 3.5, 0.2, 4.0, 6.5, 2.0, 0., 0.),
               gap_penalties_3d=(0.0, 1.75),
               feature_weights=(1., 0., 0., 0., 0., 0.), rms_cutoff=3.5,
               fit=True, surftyp=1, fit_on_first=False, gap_function=False,
               align_block=0, max_gap_length=999999, align_what='BLOCK',
               input_weights_file=None, output_weights_file=None,
               weigh_sequences=False, smooth_prof_weight=10,
               fix_offsets=(0., -1., -2., -3., -4.), substitution=False,
               comparison_type='MAT', matrix_comparison='CC',
               alignment_type='PROGRESSIVE', edit_file_ext=('.pdb', '_fit.pdb'),
               weights_type='SIMILAR', similarity_flag=False,
               bkgrnd_prblty_file='$(LIB)/blosum62_bkgrnd.prob',
               ext_tree_file=None, dendrogram_file='',
               matrix_scaling_factor=0.0069, auto_overhang=False,
               overhang_factor=0.4, overhang_auto_limit=60,
               local_alignment=False, improve_alignment=True, fit_atoms='CA',
               output='', write_whole_pdb=True, current_directory=True,
               write_fit=False, fit_pdbnam=True, rr_file='$(LIB)/as1.sim.mat',
               n_subopt=1, subopt_offset=0., align3d_trf=False,
               normalize_pp_scores=False, gap_gap_score=0.,
               gap_residue_score=0., nsegm=2, matrix_offset_3d=-0.1, io=None):
        """Align two or more proteins"""
        import salign
        if io is None:
            io = self.env.io
        (input_weights_file, read_weights) = self.__opt_file(input_weights_file)
        (output_weights_file,
         write_weights) = self.__opt_file(output_weights_file)
        if ext_tree_file is None:
            ext_tree_file = ''
        func = _modeller.mod_salign
        retval = func(self.modpt, io.modpt, self.env.libs.modpt, residue_type2,
                      no_ter, overhang, off_diagonal, matrix_offset,
                      gap_penalties_1d, gap_penalties_2d, gap_penalties_3d,
                      feature_weights, rms_cutoff, fit, surftyp,
                      fit_on_first, gap_function, align_block,
                      max_gap_length, align_what, read_weights,
                      write_weights, input_weights_file,
                      output_weights_file, weigh_sequences,
                      smooth_prof_weight, fix_offsets, substitution,
                      comparison_type, matrix_comparison,
                      alignment_type, edit_file_ext, weights_type,
                      similarity_flag, bkgrnd_prblty_file,
                      ext_tree_file, dendrogram_file, matrix_scaling_factor,
                      auto_overhang, overhang_factor,
                      overhang_auto_limit, local_alignment,
                      improve_alignment, fit_atoms, output,
                      write_whole_pdb, current_directory,
                      write_fit, fit_pdbnam, rr_file, n_subopt,
                      subopt_offset, align3d_trf, normalize_pp_scores,
                      gap_gap_score, gap_residue_score,
                      nsegm, matrix_offset_3d)
        return salign.salign_data(*retval)

    def write(self, file, alignment_format='PIR',
              alignment_features='INDICES CONSERVATION', align_block=0,
              align_alignment=False):
        """Write the alignment to a file"""
        return _modeller.mod_alignment_write(self.modpt, self.env.libs.modpt,
                                             file, alignment_format,
                                             alignment_features, align_block,
                                             align_alignment)

    def id_table(self, matrix_file):
        """Calculate percentage sequence identities"""
        from modeller.id_table import id_table
        return id_table(self, matrix_file)

    def __contains__(self, code):
        return _modeller.mod_alignment_find_code(self.modpt, code) >= 0

    def keys(self):
        return [seq.code for seq in self]

    def __delitem__(self, indx):
        ret = modutil.handle_seq_indx(self, indx,
                                      _modeller.mod_alignment_find_code,
                                      (self.modpt,))
        _modeller.mod_alnsequence_del(self.modpt, ret)

    def __getitem__(self, indx):
        ret = modutil.handle_seq_indx(self, indx,
                                      _modeller.mod_alignment_find_code,
                                      (self.modpt,))
        if type(ret) is int:
            if _modeller.mod_alnsequence_has_structure(self.modpt, ret):
                return structure(self, ret)
            else:
                return alnsequence(self, ret)
        else:
            return [self[ind] for ind in ret]

    def __get_comments(self):
        return modlist.simple_varlist(self.modpt,
                                      _modeller.mod_alignment_ncomment_get,
                                      _modeller.mod_alignment_ncomment_set,
                                      _modeller.mod_alignment_comment_get,
                                      _modeller.mod_alignment_comment_set)
    def __set_comments(self, obj):
        modlist.set_varlist(self.comments, obj)
    def __del_comments(self):
        modlist.del_varlist(self.comments)
    def __get_positions(self):
        return alnposlist(self)

    modpt = property(__get_modpt)
    comments = property(__get_comments, __set_comments, __del_comments,
                        doc="Alignment file comments")
    positions = property(__get_positions, doc="Alignment positions")


class alnsequence(sequence.sequence):
    """A single sequence within an alignment"""
    aln = None
    _num = None
    env = None

    def __init__(self, aln, num):
        self.aln = aln
        self.env = self.aln.env
        self._num = num
        sequence.sequence.__init__(self)

    def __len__(self):
        return self.nres

    def __repr__(self):
        return "Sequence %s" % repr(self.code)
    def __str__(self):
        return "<%s>" % repr(self)

    def transfer_res_prop(self):
        """Transfer residue properties of predicted secondary structure"""
        return _modeller.mod_transfer_res_prop(self.aln.modpt, self._num)

    def get_num_equiv(self, seq):
        """Get the number of identical aligned residues between this sequence
           and C{seq}."""
        neqv = 0
        for res in self.residues:
            other_res = res.get_aligned_residue(seq)
            if other_res is not None and res.type == other_res.type:
                neqv += 1
        return neqv

    def get_sequence_identity(self, seq):
        """Get the % sequence identity between this sequence and C{seq}, defined
           as the number of identical aligned residues divided by the length
           of the shorter sequence."""
        return 100.0 * self.get_num_equiv(seq) / min(len(self), len(seq))

    def __get_naln(self):
        return _modeller.mod_alignment_naln_get(self.aln.modpt)
    def __get_code(self):
        return _modeller.mod_alignment_codes_get(self.aln.modpt, self._num)
    def __set_code(self, val):
        _modeller.mod_alignment_codes_set(self.aln.modpt, self._num, val)
    def __get_atom_file(self):
        alnseq = self.__get_alnseqpt()
        return _modeller.mod_alnsequence_atom_files_get(alnseq)
    def __set_atom_file(self, val):
        alnseq = self.__get_alnseqpt()
        _modeller.mod_alnsequence_atom_files_set(alnseq, val)
    def __get_residues(self):
        return residuelist(self)
    def __get_seqpt(self):
        return _modeller.mod_alignment_sequence_get(self.aln.modpt, self._num)
    def __get_alnseqpt(self):
        return _modeller.mod_alignment_alnsequence_get(self.aln.modpt,
                                                       self._num)

    code = property(__get_code, __set_code, doc="Alignment code")
    atom_file = property(__get_atom_file, __set_atom_file, doc="PDB file name")
    naln = property(__get_naln, doc="Length of alignment (including gaps)")
    residues = property(__get_residues, doc="List of residues")
    seqpt = property(__get_seqpt)


class structure(alnsequence, coordinates):
    """A single template structure within an alignment"""

    __num = None
    __read_coord = False

    def __init__(self, aln, num):
        self.__num = num
        alnsequence.__init__(self, aln, num)
        coordinates.__init__(self)

    def __repr__(self):
        return "Structure %s" % repr(self.code)
    def __str__(self):
        return "<%s>" % repr(self)

    def __get_cdpt(self):
        if not self.__read_coord:
            aln = self.aln
            _modeller.mod_alnstructure_read(aln.modpt, self.__num,
                                            aln.env.io.modpt,
                                            aln.env.libs.modpt)
            self.__read_coord = True
        struc = _modeller.mod_alignment_structure_get(self.aln.modpt,
                                                      self.__num)
        return _modeller.mod_structure_cd_get(struc)
    cdpt = property(__get_cdpt)


class residuelist(object):
    """A list of L{alignment_residue} objects."""

    def __init__(self, seq, offset=0, length=None):
        self.seq = seq
        self.offset = offset
        self.length = length

    def __len__(self):
        if self.length is not None:
            return self.length
        else:
            return len(self.seq)

    def __getitem__(self, indx):
        ret = modutil.handle_seq_indx(self, indx)
        if type(ret) is int:
            return alignment_residue(self.seq, ret + self.offset)
        else:
            return [self[ind] for ind in ret]


class alignment_residue(sequence.sequence_residue):
    """A single residue in an aligned sequence"""

    def get_position(self):
        """Get the position in the alignment of this residue"""
        invaln = _modeller.mod_alignment_invaln_get(self.mdl.aln.modpt)
        num = _modeller.mod_int2_get(invaln, self._num, self.mdl._num)
        return self.mdl.aln.positions[num - 1]

    def get_aligned_residue(self, seq):
       """Get the residue in C{seq} that is aligned with this one, or None"""
       alnpos = self.get_position()
       return alnpos.get_residue(seq)

    def add_leading_gaps(self, ngap=1):
        """Add gaps immediately before this residue."""
        return _modeller.mod_alnresidue_add_gaps(self.mdl.aln.modpt, self._num,
                                                 self.mdl._num + 1, ngap)

    def add_trailing_gaps(self, ngap=1):
        """Add gaps immediately after this residue."""
        return _modeller.mod_alnresidue_add_gaps(self.mdl.aln.modpt,
                                                 self._num + 1,
                                                 self.mdl._num + 1, ngap)

    def remove_leading_gaps(self, ngap=1):
        """Remove gaps immediately before this residue."""
        return _modeller.mod_alnresidue_remove_gaps(self.mdl.aln.modpt,
                                                    self._num,
                                                    self.mdl._num + 1, ngap)

    def remove_trailing_gaps(self, ngap=1):
        """Remove gaps immediately after this residue."""
        return _modeller.mod_alnresidue_remove_gaps(self.mdl.aln.modpt,
                                                    self._num + 1,
                                                    self.mdl._num + 1, ngap)

    def get_leading_gaps(self):
        """Get the number of gaps in the alignment immediately preceding this
           residue."""
        mypos = self.get_position().num
        if self._num > 0:
            prepos = self.mdl.residues[self._num - 1].get_position()
            prepos = prepos.num
            return mypos - prepos - 1
        else:
            return mypos

    def get_trailing_gaps(self):
        """Get the number of gaps in the alignment immediately following this
           residue."""
        mypos = self.get_position().num
        try:
            postpos = self.mdl.residues[self._num + 1].get_position()
            postpos = postpos.num
        except IndexError:
            postpos = self.mdl.naln
        return postpos - mypos - 1


class alnposition(object):
    """An alignment position"""

    def __init__(self, aln, indx):
        self.__aln = aln
        self.__indx = indx

    def get_residue(self, seq):
       """Get the residue in C{seq} that is at this alignment position, or None
          if a gap is present."""
       aln = self.__aln
       if not isinstance(seq, alnsequence):
           raise TypeError, "Expected an 'alnsequence' object for seq"
       if seq.aln != aln:
           raise ValueError, "seq must be a sequence in the same alignment"
       ialn = _modeller.mod_alignment_ialn_get(aln.modpt)
       ires = _modeller.mod_int2_get(ialn, self.__indx, seq._num)
       if ires == 0:
           return None
       else:
           return seq.residues[ires-1]

    def __get_num(self):
        return self.__indx
    def __get_prof(self, typ):
        prof = _modeller.mod_alignment_prof_get(self.__aln.modpt)
        return _modeller.mod_float2_get(prof, self.__indx, typ)
    def __get_helix(self):
        return self.__get_prof(0)
    def __get_strand(self):
        return self.__get_prof(1)
    def __get_buried(self):
        return self.__get_prof(2)
    def __get_straight(self):
        return self.__get_prof(3)
    num = property(__get_num)
    helix = property(__get_helix, doc="Helix secondary structure")
    strand = property(__get_strand, doc="Strand secondary structure")
    buried = property(__get_buried, doc="Buriedness")
    straight = property(__get_straight, doc="Straightness")


class alnposlist(modlist.fixlist):
    """A list of L{alnposition} objects."""

    def __init__(self, aln):
        self.__aln = aln
        modlist.fixlist.__init__(self)

    def __len__(self):
        return _modeller.mod_alignment_naln_get(self.__aln.modpt)

    def _getfunc(self, indx):
        return alnposition(self.__aln, indx)
