# This file is part of ModPipe, Copyright 1997-2010 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.


import os
import modpipe.config

class FileSystem(object):
    """A ModPipe filesystem, rooted at *top*, which is either a path or a
       :class:`~modpipe.config.Config` object."""

    def __init__(self, top):
        if isinstance(top, modpipe.config.Config):
            self._top = top.DATDIR
        else:
            self._top = top

    def get_sequence_dir(self, seqid):
        return os.path.join(self._top, seqid[:3], seqid, "sequence")

    def get_model_dir(self, seqid):
        return os.path.join(self._top, seqid[:3], seqid, "models")

    def get_alignment_dir(self, seqid):
        return os.path.join(self._top, seqid[:3], seqid, "alignments")

    def get_sequence_file(self, seqid):
        return os.path.join(self.get_sequence_dir(seqid), "%s.fsa" % seqid)

    def get_selected_file(self, seqid):
        return os.path.join(self.get_sequence_dir(seqid), "%s.sel" % seqid)

    def add_sequence(self, seq, parser):
        """Add the :class:`~modpipe.sequence.Sequence` object *seq* to the
           filesystem. This sets the align code for the sequence and writes
           it to disk using *parser*, which should be one of the file parsers
           in the :class:`modpipe.sequence` module (usually
           :class:`~modpipe.sequence.FASTAFile`). The ModPipe ID for the
           sequence is returned."""
        id = seq.get_id()
        seq.code = id
        seqdir = self.get_sequence_dir(id)
        if not os.path.exists(seqdir):
            os.makedirs(seqdir)
        seqnam = self.get_sequence_file(id)
        parser.write(file(seqnam, 'w'), seq)
        return id
