# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

package PLLib::TSVModUtils;
require Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw( ScoreByTSVMod ExecTSVMod GetTSVModScore);

use PLLib::Utils;
use Cwd;
use MPLib::MPUtils;
use File::Basename;
use MPLib::Serialize;
             

use strict;

sub ScoreByTSVMod {
  #--- Get subroutine name
  my $subname = GetSubrName();

  #--- Check arguments
  my $nargs = 3;

  unless (scalar(@_) == $nargs){
     print "${subname}__E> Insufficient arguments\n" ;
     return;
  }
  my $tsvmodexe=shift@_;
  my $seq_id=shift@_;
  my $tsvmod_flag=shift@_;
  my $tmpdir=cwd();
  my $modnam= ModFileMP($seq_id);
  unless (-e $modnam) {
      warn "${subname}__E> Model data file not found: $modnam\n";
      return;
   }
   my $modlocal=basename($modnam);
      unless ( CopyFile($modnam, $modlocal) ){
      warn "${subname}__E> Could not copy model data file into current directory\n";
      warn "${subname}__E>    Source File: $modnam\n";
      warn "${subname}__E>    Target Dir : $tmpdir\n";
      return;
   }

   # Read old models file
   my $fhmod = OpenFile( $modlocal );
   my $models = ReadModelsFile($fhmod);
   close ($fhmod);

   # -- Rate each model
   for my $model (@$models) {
      # -- Get the ratings
       my $tsvmod;
       if ($tsvmod_flag ne "OFF") {
           my ($type, $rmsd, $no35,$features,$relax_count,$set_size) = GetTSVModScore($tsvmodexe,$seq_id,$model);
           if ($rmsd && $no35) {
                $tsvmod = TSVMod->new(type=>$type, 
                                      features=>$features, 
                                      predicted_rmsd=>$rmsd,
                                      predicted_no35=>$no35,
                                      relax_count=>$relax_count, 
                                      set_size=>$set_size);
           
           } else {
                $tsvmod = TSVMod->new(type=>$type);
           }
           $model->score->tsvmod($tsvmod);
       } else {
           $tsvmod = TSVMod->new(type=>"NA");
           $model->score->tsvmod($tsvmod);
       }
   }
   # -- Open a new file stream to write out the modifications
   my $modnew = (fileparse($modlocal, '\..*'))[0];
   my $fhmodnew = OpenNewFile( $modnew );
   WriteModelsFile($models, $fhmodnew);
   close ($fhmodnew);

   # -- Return new file name
   return $modnew;
}

sub GetTSVModScore {
  #--- Get subroutine name
  my $subname = GetSubrName();

  #--- Check arguments
  my $nargs = 3;

  unless (scalar(@_) == $nargs){
     print "${subname}__E> Insufficient arguments\n" ;
     return;
  }

  #-- reassign the variables
  my $tsvmodexe=shift@_;
  my $seq_id=shift@_;
  my $model=shift@_;

  my $seqdir=SeqDirMP($seq_id);
  my $alidir=AliDirMP($seq_id);
  my $moddir=ModDirMP($seq_id);
  my $alignmentfile="${alidir}/".$model->alignment->id.".ali";
  my ($method,$rmsd,$no35,$features,$set_size,$relax_count);
  my $tmpdir=cwd();
  my @files=`ls -l $moddir`;
  my $modelfile=$moddir."/".$model->id.".pdb.gz";
  ($method,$rmsd, $no35,$features,$set_size,$relax_count)=ExecTSVMod($tsvmodexe,$modelfile,$alignmentfile);
  return ($method,$rmsd, $no35,$features,$set_size,$relax_count);
  
 
}

sub ExecTSVMod {

  #--- Get subroutine name
  my $subname = GetSubrName();

  #--- Check arguments
  my $nargs = 3;

  unless (scalar(@_) == $nargs){
     print "${subname}__E> Insufficient arguments\n" ;
     return;
  }

  #-- reassign the variables
  my ( $tsvmodexe, $model, $alignment ) = @_;

  # -- Check for the NCBI executable
  unless ( -x $tsvmodexe ){
    warn "${subname}__E> Could not find TSVMod executable: $tsvmodexe\n";
    return ("NA");
  }


  # -- Run the executable and report a success/failure
  my $command="$tsvmodexe  -model $model -alignment $alignment";
  my $retval = system($command);

  # -- Check return value
  if ( $retval ){
    warn "${subname}__E> Failed executing TSVMod\n";
    warn "${subname}__E>   PROGRAM: $tsvmodexe\n";
    warn "${subname}__E>   OPTIONS: -model $model -alignment $alignment\n";
    return ("NA");
  }
  my $tsvmod=basename($model);
  $tsvmod=(split(/\./,$tsvmod))[0];

  # -- Check if output file exists
  my $outfile = "$tsvmod.tsvmod.results";
  unless ( -s $outfile ){
    warn "${subname}__W> TSVMod executable produced no (or empty) $tsvmod.tsvmod.results output file\n";
    return ("NA");
  }
  open ("OUT","$outfile");
  my ($type,$rmsd,$no35,$key,$value,$features,$relax_count,$set_size);
  while (my $line=<OUT>)  {
      chomp $line;
      ($key,$value)=split(/\s+/,$line);
      if ($key eq "Match_Type:") {
          $type=$value;
      } elsif ($key eq "Features_Used:") {
          $features=$value;
      } elsif ($key eq "Relax_count:") {
          $relax_count=$value;
      } elsif ($key eq "Set_size:") {
          $set_size=$value;
      } elsif ($key eq "Predicted_RMSD:") {
          $rmsd=$value;
      } elsif ($key eq "Pred_NO35:") {
          $no35=$value;
      }
  }
  close (OUT);
  if (!$type) {
      $type="NA";
      return ($type);
  } else {
      return ($type,$rmsd,$no35,$features,$relax_count,$set_size);
  }

}

#######################

