from __future__ import print_function
import unittest, sys, os, re
from optparse import OptionParser
import glob
import shutil
import tempfile

class _TempDir(object):
    """Simple RAII-style class to make a temporary directory"""
    def __init__(self):
        self.tmpdir = tempfile.mkdtemp()
    def __del__(self):
        shutil.rmtree(self.tmpdir, ignore_errors=True)


# Only use coverage if it's new enough
try:
    import coverage
    if not hasattr(coverage.coverage, 'combine'):
        coverage = None
except ImportError:
    coverage = None

class RunAllTests(unittest.TestProgram):
    """Custom main program that also displays a final coverage report"""
    def __init__(self, html_coverage, *args, **keys):
        self.html_coverage = html_coverage
        if coverage:
            # Start coverage testing now before we import any modules
            cwd = os.path.dirname(sys.argv[0])
            self.topdir = os.path.abspath(os.path.join(cwd, '..', '..'))
            self.mods = glob.glob('%s/lib/python/modpipe/*.py' % self.topdir)
            for dir in ('main', 'scripts'):
                self.mods.extend(glob.glob('%s/lib/python/modpipe/%s/*.py' \
                                           % (self.topdir, dir)))
            for dir in ('main', 'python', 'src'):
                self.mods.extend(glob.glob('%s/%s/int/*.py' \
                                           % (self.topdir, dir)))
            self.cov = coverage.coverage(branch=True, include=self.mods)
            self.cov.start()
            self._setup_subprocess_coverage()

        # Run the tests
        unittest.TestProgram.__init__(self, *args, **keys)

    def _setup_subprocess_coverage(self):
        # Make sure that command line tools are also covered as soon as they
        # import the modpipe package. We do this by making a special
        # sitecustomize.py. runscript.py adds this directory (via
        # the MODPIPE_COVERAGE environment variable) to the Python path
        # so it gets loaded before anything else.
        self.__tmpdir = t = _TempDir()
        os.environ['MODPIPE_COVERAGE'] = t.tmpdir
        f = open(os.path.join(t.tmpdir, 'sitecustomize.py'), 'w')
        f.write("import coverage\n")
        f.write("__cov = coverage.coverage(branch=True, data_suffix=True, "
                "auto_data=True)\n")
        f.write("__cov.start()\n")

    def runTests(self):
        self.testRunner = unittest.TextTestRunner(verbosity=self.verbosity)
        result = self.testRunner.run(self.test)

        if coverage:
            self.cov.stop()
            self.cov.combine()
            self.cov.use_cache(False)
            print("\nPython coverage report\n", file=sys.stderr)

            # Don't show full paths to modules in coverage output
            if hasattr(coverage.files, 'RELATIVE_DIR'):
                coverage.files.RELATIVE_DIR = self.topdir + '/'
            else:
                self.cov.file_locator.relative_dir = self.topdir + '/'
            self.cov.report(self.mods, file=sys.stderr)
            html = self.html_coverage
            if html:
                self.cov.html_report(self.mods,
                                     directory=os.path.join(html, 'python'))
            for cov in glob.glob('.coverage.*'):
                os.unlink(cov)
        sys.exit(not result.wasSuccessful())


def regressionTest():
    path = os.path.abspath(os.path.dirname(sys.argv[0]))
    files = os.listdir(path)
    test = re.compile("^test_.*\.py$", re.IGNORECASE)
    files = filter(test.search, files)
    modnames = [os.path.splitext(f)[0] for f in files]

    modobjs = [__import__(m) for m in modnames]
    tests = [unittest.defaultTestLoader.loadTestsFromModule(o) for o in modobjs]
    return unittest.TestSuite(tests)

def parse_options():
    parser = OptionParser()
    parser.add_option("-v", dest="verbose", action='store_true',
                      help="verbose test output")
    return parser.parse_args()

if __name__ == "__main__":
    opts, args = parse_options()
    sys.argv = [sys.argv[0]]
    if opts.verbose:
        sys.argv.append('-v')
    if len(args) == 1:
        html_coverage = args[0]
    else:
        html_coverage = None

    RunAllTests(html_coverage, defaultTest="regressionTest")
