# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

package MPLib::MPUtils;
require Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw( SeqDirMP SeqFileMP PrfFileMP AliDirMP CopyNRSEQDB 
              CopySEQDB CopyTEMPLATESEQDB CountHits HitFileMP 
              AliFileMP CopyAlignmentMP ModDirMP CopyXPRFDB
              PsiBlastPrfFileMP PsiBlastChkFileMP CopyNCBISEQDB
              ReadUNQ WriteSGEMP WriteSGETB ModFileMP CleanupTMP 
              SortMPDataByLength SelFileMP FindTemplate HHSuitePrfFileMP
            );
              


use strict;
use PLLib::Utils;
use MPLib::MPInit;
use MPLib::Binaries;

sub SelFileMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $seqdir = SeqDirMP( $md5id );
   my $selnam = "${seqdir}/${md5id}.sel";

   # --- Return
   return $selnam;
}

sub hit_length {
  my ($hit) = @_;
  return $hit->region->[1] - $hit->region->[0] + 1;
}

sub SortMPDataByLength {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # -- Reassing input arguments
   my $datl = $_[0];

   # -- Sort models by length
   my @sorted_datl = sort { hit_length($b) <=> hit_length($a) } @$datl;

   # -- Return sorted array
   return \@sorted_datl;
}


sub CleanupTMP {

   use File::Path;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $tmpdir ) = $_[0];

   unless ( rmtree( $tmpdir ) ){
      warn "${subname}__E> Could not remove temporary directory\n";
      return;
   }

   # --- Return
   return 1;
}

sub ModFileMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $seqdir = SeqDirMP( $md5id );
   my $modnam = "${seqdir}/${md5id}.mod";

   # --- Return
   return $modnam;
}

sub WriteSGEMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $min_nargs = 14;

   unless ( scalar(@_) >= $min_nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my $sgefh = $_[0];
   my $logdir = $_[1];
   my $jobname = $_[2];
   my $modpipe = $_[3];
   my $disks = $_[4];
   my $nodes = $_[5];
   my $priority = $_[6];
          
   my $conffile = $_[7];
   my $exitstage = $_[8];
   my $hitsmode = $_[9];
   my $evaluehits = $_[10];
   my $cleanup = $_[11];
   my $tsvmod_flag=$_[12];
   my $final_modby = $_[13];
          
   my $ntasks = $_[14];
   my $tasks = $_[15];
   my $tmpdir = $_[16];

   # --- Optional last two arguments, for template based modeling

   my $template = $_[17]||"NONE";
   my $template_option=$_[18]||"ALL";

   # -- Format the task list
   my $taskstr = join(" ", @$tasks);

   # -- Format the sge job file
   my $sgestr = undef;
   if ($final_modby && ($final_modby ne "")) {
       $final_modby="--final_models_by ${final_modby} ";
   } else {
     $final_modby="";
   }
   if ($tsvmod_flag eq "OFF") {
       $tsvmod_flag="--score_by_tsvmod OFF";
   } else {
       $tsvmod_flag="";
   }
   my $cleanflag="";
   if ($tmpdir && ($cleanup eq "ON")) {
       $cleanflag="rm -r $tmpdir";
   }
       
   $sgestr = qq{
         #!/bin/csh
         #\$ -S /bin/csh
         #\$ -cwd
         #\$ -o $logdir
         #\$ -e $logdir
         #\$ -j y
         #\$ -l $disks
         #\$ -l $nodes
         #\$ -r y
         #\$ -N $jobname
         #\$ -p $priority
         #\$ -t 1-$ntasks
 
         set tasks=(${taskstr})
         set input=\$tasks[\$SGE_TASK_ID]
 
         echo Running on: `hostname`
         echo Started at: `date`
         echo
 
         $modpipe --conf_file $conffile \\
                                        --sequence_id \${input} \\
                                        --hits_mode $hitsmode \\
                                        --exit_stage $exitstage \\
                                        --evalue_hits $evaluehits \\
                                        --clean_up $cleanup \\
                                        $final_modby \\
                                        $tsvmod_flag \\
                                        --template $template \\
                                        --template_option $template_option
 
         echo
         $cleanflag/\${input}
         echo Completed at: `date`
               };

   # -- Reformat the string and write to file
   $sgestr =~ s/^\n//;
   $sgestr =~ s/^[^\S\n]{9}//gm;
   print $sgefh $sgestr, "\n";
   
   # --- Return value
   return 1;
}

sub WriteSGETB {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 16;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments: ".scalar(@_) ." instead of $nargs\n";
      return;
   }

   # --- Reassign input arguments
   my $sgefh = $_[0];
   my $runname = $_[1];
   my $rundir = $_[2];
   my $nodes = $_[3];
   my $nodes_tb = $_[4];
   my $disks = $_[5];
   my $modpipeclusterbase = $_[6];
   my $finalmodby = $_[7];
   my $hitsmode = $_[8];
   my $pdbid = $_[9];
   my $pdbchain = $_[10];
   my $template_option = $_[11];
   my $tmpdir = $_[12];
   my $pdb_file = $_[13];
   my $tsvmod_flag = $_[14];
   my $max_sequences = $_[15]||30000;
   my $max_concurrent_tasks_tb = $_[16]||9999;
   
   my $templatebased = "$modpipeclusterbase/main/TemplateBased.pl";
   my $modpipe = "$modpipeclusterbase/main/ModPipe.pl";

   # -- Format the sge job file
   my $sgestr = undef;
   if ($pdbchain && ($pdbchain ne "")) {
       $pdbchain="--pdb_chain $pdbchain";
   } else {
       $pdbchain="";
   }
   my $cleanflag="";
   if ($tmpdir) {
      $cleanflag="rm -r $tmpdir/$pdbid";
   }
   if ($pdb_file) {
       $pdb_file="--pdb_file $pdb_file";
   } else {
       $pdb_file="";
   }
   if (!$tsvmod_flag || ($tsvmod_flag eq "")) {
       $tsvmod_flag="ON";
   }
   $sgestr = qq{
         #!/bin/csh
         #\$ -S /bin/csh
         #\$ -cwd
         #\$ -o $rundir
         #\$ -e $rundir
         #\$ -j y
         #\$ -l $disks
         #\$ -l $nodes_tb
         #\$ -tc $max_concurrent_tasks_tb
         #\$ -r y
         #\$ -N $runname
         #\$ -p 0
 
         echo Running on: `hostname`
         echo Started at: `date`
         echo
 
         # nodes and disks arguments for ModPipe job.

         $templatebased --pdb_code $pdbid \\
                        $pdbchain  \\
                        $pdb_file  \\
                        --template_option $template_option \\
                        --job_name $runname \\
                        --runname $runname \\
                        --run_directory $rundir \\
                        --hits_mode $hitsmode \\
                        --final_models_by $finalmodby \\
                        --nodes $nodes \\
                        --score_by_tsvmod $tsvmod_flag \\
                        --tmpdir_sequence $tmpdir \\
                        --tmpdir_profile $tmpdir \\
                        --disks $disks
 
         echo

         $cleanflag

         echo Completed at: `date`
               };

   # -- Reformat the string and write to file
   $sgestr =~ s/^\n//;
   $sgestr =~ s/^[^\S\n]{9}//gm;
   print $sgefh $sgestr, "\n";
   
   # --- Return value
   return 1;
}

sub ReadUNQ {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # -- Get input argument
   my $unqfile = $_[0];

   # -- Check if unq file exists
   return unless ( -e $unqfile );

   # -- Read in the unq file
   my @ids = my @names = ();
   my $unqfh = OpenFile( $unqfile );
   while ( my $line = <$unqfh> ){
      chomp $line;
      my ($id, $name) = (split(" ", $line))[0,2];
      push @ids, $id; 
      push @names, $name;
   }
   close( $unqfh );

   # -- Return
   return(\@ids, \@names);
}

sub CopyNCBISEQDB{

use Cwd;
use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 0;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Get current directory
   my $currdir = cwd();

   # --- Copy the database over; handle split into multiple files
   foreach my $blsext ( qw( phr psq pin ) ){
      my @seqdbs = glob("${init::ncbiseqdb}*.${blsext}");
      if (scalar @seqdbs == 0) {
         warn "${subname}__E> Could not find sequence database\n";
         warn "${subname}__E>    Looked in ${init::ncbiseqdb}*.${blsext}\n";
         return;
      }

      for my $ncbiseqdb (@seqdbs) {
         my $ncbiseqdblocal = basename($ncbiseqdb);

         unless ( CopyFile(${ncbiseqdb}, ${ncbiseqdblocal}) ){
            warn "${subname}__E> Failed copying non-redundant sequence database\n";
            warn "${subname}__E>    Source File: ${ncbiseqdb}\n";
            warn "${subname}__E>    Target Dir : $currdir\n";
            return;
         }
      }
   }

   # --- Copy the database alias file over, if one exists
   my $ncbiseqdblocal = basename(${init::ncbiseqdb});
   if ( -e "${init::ncbiseqdb}.pal" 
        && ! CopyFile("${init::ncbiseqdb}.pal", 
                      "${ncbiseqdblocal}.pal" ) ){
         warn "${subname}__E> Failed copying non-redundant sequence database\n";
         warn "${subname}__E>    Source File: ${init::ncbiseqdb}.pal\n";
         warn "${subname}__E>    Target Dir : $currdir\n";
   }

   # --- Return success
   return $ncbiseqdblocal;
}

sub PsiBlastChkFileMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $seqdir = SeqDirMP( $md5id );
   my $chknam = "${seqdir}/${md5id}-psiblast-${init::nrdbtag}.chk";

   # --- Return
   return $chknam;
}

sub HHSuitePrfFileMP {

use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 4;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $hhsuite, $md5id, $profile, $type ) = @_;

   my $seqdir = SeqDirMP( $md5id );
   my $dbtype;

   my ($database_file, $dbtag, $prfa3m, $prfhhr);

   print "${subname}__C> Starting HHsuite $hhsuite type $type \n";
   if ($hhsuite eq "hhblits") {
      if ($type eq "uniprot20") {
         $database_file = $init::hhblitsseq;
      } elsif ($type eq "pdb70") {
         $database_file = $init::hhblitspdb;
      }

      $prfa3m = "${seqdir}/${md5id}-hhblits-${type}.a3m";
      $prfhhr = "${seqdir}/${md5id}-hhblits-${type}.hhr";

   } elsif ($hhsuite eq "hhsearch") {
   
      if ($type eq "uniprot20") {
         $database_file = $init::hhsearchseq;
      } elsif ($type eq "pdb70") {
         $database_file = $init::hhsearchpdb;
      }
      $prfa3m = "${seqdir}/${md5id}-hhsearch-${type}.a3m";
      $prfhhr = "${seqdir}/${md5id}-hhsearch-${type}.hhr";
   
   }

   # --- Return
   return ($database_file, $prfa3m, $prfhhr);

}

sub PsiBlastPrfFileMP {

use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $seqdir = SeqDirMP( $md5id );
   my $dbtag = (fileparse(${init::ncbiseqdb},'\..*'))[0];
   my $prfnam = "${seqdir}/${md5id}-psiblast-${dbtag}.pir";

   # --- Return
   return $prfnam;
}

sub CopyXPRFDB{

use Cwd;
use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 0;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Get current directory
   my $currdir = cwd();

   # --- Copy the list of structure profiles over
   my $xprflist = basename( $init::xprflist );
   unless ( CopyFile($init::xprflist, $xprflist) ){
      warn "${subname}__E> Failed copying structure profile database\n";
      warn "${subname}__E>    Source File: $init::xprflist\n";
      warn "${subname}__E>    Target Dir : $currdir\n";
      return;
   }

   # --- Copy the database of structure profile PSSMs
   my $xprfpssmdb = basename( $init::xprfpssmdb );
   unless ( CopyFile($init::xprfpssmdb, $xprfpssmdb) ){
      warn "${subname}__E> Failed copying structure profile database\n";
      warn "${subname}__E>    Source File: $init::xprfpssmdb\n";
      warn "${subname}__E>    Target Dir : $currdir\n";
      return;
   }

   # -- Open the list file
   my $fh_xprflist = OpenFile($xprflist);
   
   # -- Read in all the lines into array
   chomp(my @xprfs = <$fh_xprflist>);

   # --- Report the number of structure profiles
   my $numxprfs = scalar( @xprfs );
   print "${subname}__M> No. of structure profiles in database: $numxprfs\n";
   close($fh_xprflist);

   # --- Return success
   return ($init::xprflist, $init::xprfpssmdb);
}


sub ModDirMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $subdir = substr($md5id, 0, 3);
   my $moddir = "${init::datdir}/${subdir}/${md5id}/models";

   # --- Return
   return $moddir;
}

sub CopyAlignmentMP {

use Cwd;
use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 2;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Get current directory
   my $currdir = cwd();

   # --- Re-assign input arguments
   my ($seqid, $aliid) = @_;

   # --- Make sure alignment exists
   my $alinam = AliFileMP( $seqid, $aliid );
   unless ( -e $alinam ){
      warn "${subname}__E> Alignment not found: $alinam\n";
      return;
   }

   # --- Copy the alignment to the current directory
   my $alilocal = basename( $alinam );
   unless ( CopyFile($alinam, $alilocal) ){
      warn "${subname}__E> Failed copying alignment\n";
      warn "${subname}__E>    Source File: $alinam\n";
      warn "${subname}__E>    Target Dir : $currdir\n";
      return;
   }

   # --- Return sucess
   return $alilocal;
}

sub AliFileMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 2;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $seqid, $aliid ) = @_;

   # --- Create the sequence directory
   my $alidir = AliDirMP( $seqid );
   my $alinam = "${alidir}/${aliid}.ali";

   # --- Return
   return $alinam;
}

sub HitFileMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $seqdir = SeqDirMP( $md5id );
   my $hitnam = "${seqdir}/${md5id}.hit";

   # --- Return
   return $hitnam;
}

sub CountHits{

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Assign input variables
   my $hitfile = $_[0];

   # --- Check if hitfile is present
   unless ( -e $hitfile ){
      warn "${subname}__E> File not found: $hitfile\n";
      return;
   }

   # --- Get the lines corresponding to hits
   my @hitlines = ();
   my $fh_hit = OpenFile( $hitfile );
   while ( my $line = <$fh_hit> ){
      next if ( $line =~ /^#/ );
      chomp $line;
      push @hitlines, $line;
   }
   close($fh_hit);

   # --- Return hit lines
   return @hitlines;
}

sub FindTemplate{

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 3;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Assign input variables
   my $hits = $_[0];
   my $template = $_[1];
   my $template_option = $_[2];

   # --- Option for filtering the hits and removing other 
   #     templates is still missing here

   # --- See if template of interest is among hits
   foreach my $hit ( @$hits ) {
      foreach my $template_hit (@{$hit->templates}) {
        if ($template_hit->code . $template_hit->chain eq $template) {
          return $hit;
        }
      }
    }
}

sub CopyTEMPLATESEQDB{

use Cwd;
use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 0;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Get current directory
   my $currdir = cwd();

   # --- Copy the database over
   my $tmpseqdblocal = basename( $init::templateseqdb );
   unless ( CopyFile($init::templateseqdb, $tmpseqdblocal) ){
      warn "${subname}__E> Failed copying non-redundant sequence database\n";
      warn "${subname}__E>    Source File: $init::templateseqdb\n";
      warn "${subname}__E>    Target Dir : $currdir\n";
      return;
   }

   # --- Return success
   return $tmpseqdblocal;
}


sub CopyNRSEQDB{

use Cwd;
use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 0;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Get current directory
   my $currdir = cwd();

   # --- Copy the database over
   my $nrseqdblocal = basename( $init::nrseqdb );
   unless ( CopyFile($init::nrseqdb, $nrseqdblocal) ){
      warn "${subname}__E> Failed copying non-redundant sequence database\n";
      warn "${subname}__E>    Source File: $init::nrseqdb\n";
      warn "${subname}__E>    Target Dir : $currdir\n";
      return;
   }

   # --- Return success
   return $nrseqdblocal;
}

sub CopySEQDB{

use Cwd;
use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   # --- Reassign input arguments
   my $seqdb = $_[0];

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Incorrect number of arguments\n";
      return;
   }

   # --- Get current directory
   my $currdir = cwd();

   # --- Copy the database over
   my $seqdblocal = basename( $seqdb );
   unless ( CopyFile($seqdb, $seqdblocal) ){
      warn "${subname}__E> Failed copying sequence database\n";
      warn "${subname}__E>    Source File: $seqdb\n";
      warn "${subname}__E>    Target Dir : $currdir\n";
      return;
   }

   # --- Return success
   return $seqdblocal;
}

sub AliDirMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $subdir = substr($md5id, 0, 3);
   my $seqdir = "${init::datdir}/${subdir}/${md5id}/alignments";

   # --- Return
   return $seqdir;
}


sub PrfFileMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $min_nargs = 1;

   unless ( scalar(@_) >= $min_nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Optional second argument - replaces configuration NRDBTAG
   my $nrdbtag = $init::nrdbtag;
   if ( scalar(@_) > 1 ) {
      $nrdbtag = $_[1];
   }

   # --- Create the sequence directory
   my $seqdir = SeqDirMP( $md5id );
   my $prfnam = "${seqdir}/${md5id}-$nrdbtag.prf";

   # --- Return
   return $prfnam;
}

sub SeqFileMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $seqdir = SeqDirMP( $md5id );
   my $seqnam = "${seqdir}/${md5id}.fsa";

   # --- Return
   return $seqnam;
}

sub SeqDirMP {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ( $md5id ) = $_[0];

   # --- Create the sequence directory
   my $subdir = substr($md5id, 0, 3);
   my $seqdir = "${init::datdir}/${subdir}/${md5id}/sequence";

   # --- Return
   return $seqdir;
}

