# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.


from modpipe.miscutils import *

class Matrix2D(object):
    """Simple 2D matrix class. Once an object is created, it can be assigned
       to or read using the standard [] operator, passing a row and column
       (0-based). For example::

           m = Matrix2D(5,5)
           print m[0,0]
           m[4,0] = 'test'
    """

    def __init__(self, rows, cols):
        self.rows = rows
        self.cols = cols
        self.matrix = []
        for i in range(rows):
            ith_row = []
            for j in range(cols):
                ith_row.append(0)
            self.matrix.append(ith_row)

    def __setitem__(self, rowcol, val):
        row, col = rowcol
        self.matrix[row][col] = val

    def __getitem__(self, rowcol):
        row, col = rowcol
        return self.matrix[row][col]

    def __get_maxlen(self):
        max_len = 0
        for i in range(self.rows):
            for j in range(self.cols):
                ele = self[i,j]
                ele_len = len(str(ele))
                if ele_len > max_len:
                    max_len = ele_len
        return max_len

    def write(self, file=None, dimensions=True):
        """Write out to the named file. If *dimensions* is *True*, also write
           the number of rows and columns in the matrix. The file format is
           suitable for input to Modeller methods that read matrices."""
        max_len = self.__get_maxlen()
        fh = open_output_stream(file)
        if dimensions is True:
            outstr = "%d %d\n" % (self.rows, self.cols)
            fh.write(outstr)
        for i in range(self.rows):
            fh.write(' '.join([str(x).rjust(max_len) for x in self.matrix[i]]))
            fh.write('\n')
