# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

package PLLib::CEUtils;
require Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw(ExecCE CETOPIR ); 
             


use strict;
use PLLib::Utils;

sub ExecCE {

   #--- Get subroutine name
   my $subname = GetSubrName();

   #--- Check arguments
   my $nargs = 7;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   #--- Get the input argument
   my ( $ceexec, $pdb1, $chn1, $pdb2, $chn2,
        $tmpdir, $mkdbdir ) = @_;

   # -- Run the CE program
   my @ceout = 
     `${ceexec} - ${pdb1} ${chn1} ${pdb2} ${chn2} ${tmpdir} ${mkdbdir} 2>&1`;

   # -- Check the output
   my $alicnt = 0; my $tmatcnt = 0;
   foreach my $line ( @ceout ){
     $alicnt++ if ( $line =~ /^Alignment length/ );
     $tmatcnt++ if ( $line =~ /^\s*X2 =.*/ );
   }

   # -- Return failure if counts do not match
   return if ( $alicnt != $tmatcnt || $alicnt == 0 );

   # -- Return log lines as sucess
   return ( \@ceout );
}

sub CETOPIR {

   use PLLib::Sequence;
   use PLLib::PDBUtils;

   #--- Get subroutine name
   my $subname = GetSubrName();

   #--- Check arguments
   my $nargs = 7;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   #--- Get the input arguments
   my ($ceali, $acode1, $pdb1, $chn1, $acode2, $pdb2, $chn2) = @_;

   # --- Parse the CE output
   my ($seq1, $seq2) = undef;
   my $outpir = undef;
   foreach my $ln ( @$ceali ){
     chomp $ln;

     # --- Get the Alignment details
     if ( $ln =~ /^Alignment length/ ){
        $ln =~ s/CPU/\nC;CPU/;
        my $tmpstr = sprintf("C;%s\n", $ln);
        $outpir = $tmpstr;
     }

     # --- Get the sequence lines of chain 1
     if ( $outpir && $ln =~ /^Chain 1:/ ){
        my $seqtmp = (split(" ", $ln))[3];
        $seq1 .= $seqtmp;
     }

     # --- Get the sequence lines of chain 2
     if ( $outpir && $ln =~ /^Chain 2:/ ){
        my $seqtmp = (split(" ", $ln))[3];
        $seq2 .= $seqtmp;
     }

     last if ( $ln =~ /\s+X2.*X1.*Y1.*Z1.*/ );
   }

   # -- Get the reference sequences
   my ( $aaseq1, $aaidx1) = undef;
   unless ( ( $aaseq1, $aaidx1 ) = GetChnSeqIdx( $pdb1, $chn1 )){
     warn "${subname}__E> Failed getting reference sequence: \"$chn1\", $pdb1\n";
     return;
   }

   my ( $aaseq2, $aaidx2 ) = undef;
   unless ( ( $aaseq2, $aaidx2 ) = GetChnSeqIdx( $pdb2, $chn2 ) ){
     warn "${subname}__E> Failed getting reference sequence: \"$chn2\", $pdb2\n";
     return;
   }

   # -- Get the beg/end of sequences
   my ( $beg1, $end1, $beg2, $end2 );
   my $seq1_copy = RemoveDashes( $seq1 );
   unless ( ($beg1, $end1) = FindSeqInSeq($seq1_copy, $aaseq1) ){
     warn "${subname}__E> Sequence in CE alignment and PDB do not match: $pdb1\n";
     return;
   }
   
   my $seq2_copy = RemoveDashes( $seq2 );
   unless ( ($beg2, $end2) = FindSeqInSeq($seq2_copy, $aaseq2) ){
     warn "${subname}__E> Sequence in CE alignment and PDB do not match: $pdb2\n";
     return;
   }
   
   # --- Format the sequences
   #     Decrement the beg1/end1 & beg2/end2 numbers by 1. This
   #     is because the indexing routine returns the positions
   #     starting at 1 (real world).
   my $outpir1 = WritePIR($acode1, "structureX", $pdb1, 
                          $aaidx1->[$beg1-1], $chn1, 
                          $aaidx1->[$end1-1], $chn1,
                          $acode1, "CE Alignment", -1.00, 0.0,
                          $seq1, 75);
   
   my $outpir2 = WritePIR($acode2, "structureX", $pdb2, 
                          $aaidx2->[$beg2-1], $chn2, 
                          $aaidx2->[$end2-1], $chn2,
                          $acode2, "CE Alignment", -1.00, 0.0,
                          $seq2, 75);

   # --- Concatenate the lines
   $outpir .= $outpir1 . $outpir2;

   return($outpir);
}


