# This file is part of ModPipe, Copyright 1997-2020 Andrej Sali
#
# ModPipe is free software: you can redistribute it and/or modify
# it under the terms of version 2 of the GNU General Public License
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with ModPipe.  If not, see <http://www.gnu.org/licenses/>.

package PLLib::Utils;
require Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw( OpenFile OpenNewFile TMPName MakeDirSys MakeDir GetSubrName
              CopyFile GetFileSize MakeTMPDir
              GetArch CompressFile GetListIntSect UnCompressFile
              CleanUp GetTime FetchLines GetDirList OpenFileLock ReadConf
              Log TryFileLock LockFileHandle IndexInList
              ComplementArray MakeLocalTMPDir CheckArgs);


use strict;
use File::Basename;
use File::Path;
use Carp;

sub CheckArgs {

   # -- Get name
   my $subname = GetSubrName();

   my $nargs = 3;

   # -- Check the options for this subroutine
   unless ( scalar(@_) == $nargs ){
      print "${subname}__E> Insufficient arguments\n";
      return;
   }

   # -- Get the arguments
   my ( $name, $args1, $args2 ) = @_;

   # -- Now check the arguments for the specified subroutine
   unless ( $args1 == $args2 ){
      print "${name}__E> Insufficient Arguments:\n";
      print "${name}__E>    Expected : $args1\n";
      print "${name}__E>    Specified: $args2\n";
      die;
   }

   return 1;
}

sub MakeLocalTMPDir {

   use IO::File;
   use File::Temp qw/ tempdir /;
   use File::Path;

   #--- Get subroutine name
   my $subname = GetSubrName();

   #--- Check arguments
   my $nargs = 0;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   my $tmpdir = tempdir();

   # --- return the tmpdir name
   return( $tmpdir );
}



# -- This routine takes two arrays and returns the array equivalent
#    of A - B, ie., all elements in A that are not in B.

sub ComplementArray {

   # -- Get subroutine name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 2;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   # -- Get the two lists
   my ( $lista, $listb ) = @_;

   # -- build lookup table of second array
   my %lookup;
   @lookup{@$listb} = ();

   # -- Now scan each element of list A
   my @complement = ();
   foreach my $a ( @$lista ){
      push @complement, $a unless ( exists $lookup{$a} );
   }

   return @complement;
}



sub IndexInList {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 2;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($array, $element) = @_;

   # --- Ensure that the array is not empty
   return unless ( scalar(@$array) );

   # --- Run through the array to locate the element
   my $idx = 0;
   foreach my $elem ( @$array ){
      $idx++;
      last if ( $elem eq $element );
   }

   # -- Decrement the counter to account for a zero start
   $idx--;

   # -- Return array position
   return $idx;
}


sub LockFileHandle {

   use Fcntl qw/:flock/;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return ;
   }

   # --- Reassign input arguments
   my $fh = $_[0];

   # --- Check for presence
   return unless ( $fh );

   # -- Try to get a lock on the file
   my $attempts = 0;
   until ( flock( $fh, LOCK_EX|LOCK_NB ) ){
      sleep 5;
      $attempts++;

      # -- Exit if lock is not obtained after 10 attempts.
      if ( $attempts > 10 ){
         warn "${subname}__E> Failed getting a lock on file\n";
         warn "${subname}__E> Attempts: $attempts\n";
         return;
      }
   }

   return $fh;
}


sub TryFileLock {

   use IO::File;
   use Fcntl qw(:DEFAULT :flock);

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 2;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return ;
   }

   # --- Reassign input arguments
   my ($filename, $mode) = @_;

   # -- Check for mode
   unless ( $mode eq "r" || $mode eq "w" || $mode eq "a" ){
      warn "${subname}__E> Improper mode specified for access: $mode\n";
      return;
   }

   # --- Check for presence
   if ( $mode eq "r" && ! -e $filename ){
      print "${subname}__W> Could not file $filename\n";
      return;
   }

   # --- Open file and return filehandle
   my $fh = IO::File->new($filename, $mode);

   # -- Try to get a lock on the file
   unless ( flock( $fh, LOCK_EX|LOCK_NB ) ){
      warn "${subname}__E> File locked by another process: $filename\n";
      return;
   }

   return $fh;
}

sub Log {

   use Fcntl qw/:flock/;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 2;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # -- Get input argument
   my ($fhlog, $messagestr) = @_;

   # -- Open a locked version of the file
   my $fhlock = LockFileHandle( $fhlog )
      or return;

   # -- Format message
   $messagestr =~ s/^\n//; # -- Remove empty lines
   $messagestr =~ s/^[^\S\n]{1,}//gm;
   $messagestr =~ s/\n/ /g;

   # -- Log message
   my $subtime = scalar localtime;
   print $fhlock "$subtime : $messagestr\n";

   # -- Release file lock
   flock($fhlock, LOCK_UN);

   # -- Return
   return 1;
}


sub ReadConf {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my $conffile = $_[0];

   # --- Check for existence of configuration file
   unless ( -e $conffile ){
      print "${subname}__D> Could not find configuration file $conffile\n";
      return;
   }

   # --- Read in the configuration file
   my %conf = ();
   my ($tag, $value) = undef;
   my $fh_conf = OpenFile( $conffile );
   while ( my $line = <$fh_conf> ){
      chomp $line;
      next if ( $line =~ /^#/ || $line =~ /^[\s]*$/ );
      ($tag, $value) = split(" ", $line);

      # -- Expand (interpolate) any environment variables specified
      #    See PERL COOKBOOK Recipe 1.8
      $value =~ s{
                   \$
                  (\w+)
                 }{
                    if ( defined $ENV{$1} ){
                      $ENV{$1};
                    } else {
                       "[NO VARIABLE: \$$1]";
                      }
                  }egx;


      # -- Return if something goes wrong or if
      #    a suitable variable not found
      if ( $value =~ /NO VARIABLE/ || $value =~ /\$/){
         warn "${subname}__E> Do not understand variable:\n";
         warn "${subname}__E>   CONFIGURATION LINE: $line\n";
         return;
      }

      # -- Now assign the value to the hash
      $conf{$tag} = $value;
   }
   close($fh_conf);

   # --- Return hash
   return \%conf;
}

sub OpenFileLock {

   use IO::File;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 2;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return ;
   }

   # --- Reassign input arguments
   my ($filename, $mode) = @_;

   # -- Check for mode
   unless ( $mode eq "r" || $mode eq "w" || $mode eq "a" ){
      warn "${subname}__E> Improper mode specified for access: $mode\n";
      return;
   }

   # --- Check for presence
   if ( $mode eq "r" && ! -e $filename ){
      print "${subname}__W> Could not open file $filename\n";
      return;
   }

   # --- Open file and return filehandle
   my $fh = IO::File->new($filename, $mode);

   # -- Try to get a lock on the file
   my $attempts = 0;
   until ( flock( $fh, 2|4 ) ){
      sleep 5;
      $attempts++;

      # -- Exit if lock is not obtained after 10 attempts.
      if ( $attempts > 10 ){
         warn "${subname}__E> Failed getting a lock on file\n";
         warn "${subname}__E> File    : $filename\n";
         warn "${subname}__E> Attempts: $attempts\n";
         return;
      }
   }

   return $fh;
}

sub GetDirList {

   # -- Get name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # -- Get the arguments
   my $dir = $_[0];

   # -- Get the list of files in folder
   unless ( opendir(DIR, $dir) ){
      warn "${subname}__E> Could not open folder: $dir\n";
      return;
   }

   my @files = ();
   while ( my $file = readdir(DIR) ){
      next if ( $file =~ /^\..*/ );
      push @files, $file;
   }
   closedir(DIR);

   # -- return
   return @files;
}

sub FetchLines {

   # -- Get subroutine name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 1;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   # -- Get file argument list
   my $filename = $_[0];

   # -- Check if file exists
   return [ ] unless ( -e $filename );

   # -- Open the file and read in the lines
   my $fhfile = OpenFile( $filename );
   my @lines = ();
   chomp(@lines = <$fhfile>);
   close($fhfile);

   # -- Return reference to file contents
   return \@lines;
}



sub GetTime {

   # -- Get subroutine name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 0;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   my $time = `date`;
   my $host = `hostname`;
   chomp($time);
   chomp($host);
   my $msg = $host . " : " . $time;

   return $msg;
}

sub CleanUp {

   # -- Get subroutine name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 1;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   # -- Get file argument list
   my $filelist = $_[0];

   # -- Remove files
   foreach my $file ( @$filelist ){
      unlink( $file ) if ( -e $file );
   }

   # -- Return
   return 1;
}


sub UnCompressFile {

   use File::Basename;

   # -- Get subroutine name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 1;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   # -- Get file name
   my $filen = $_[0];

   # -- Check if file handle is defined
   unless ( -e $filen ){
      warn "${subname}__E> File not found: $filen\n";
      return;
   }

   # -- Does file have .gz extension
   my ($name, $dir, $ext) = fileparse($filen, '\.gz');
   unless ( $ext eq ".gz" ){
      warn "${subname}__E> File does not have .gz extension: $filen\n";
      return;
   }

   # -- Check for gunzip
   unless ( `which gunzip` ){
      warn "${subname}__E> gunzip not found\n";
      return;
   }

   # -- UnCompress file
   if ( `gunzip -f $filen` ){
      warn "${subname}__E> Failed un-compressing file: $filen\n";
      return;
   }

   # -- Check for un-compressed file
   my $ungzfile = "${dir}/${name}";
   unless ( -e $ungzfile ){
      warn "${subname}__E> Could not find un-compressed file: $ungzfile\n";
      return;
   }

   # -- Return ungz filename
   return $ungzfile;
}



sub GetListIntSect {

   # -- Get subroutine name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 2;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   # -- Get the two lists
   my ( $lista, $listb ) = @_;

   # -- build lookup table of second array
   my %lookup;
   @lookup{@$listb} = ();

   # -- Now scan each element of list A
   my @intsect = ();
   foreach my $a ( @$lista ){
      push @intsect, $a if ( exists $lookup{$a} );
   }

   return @intsect;
}


sub CompressFile {

   # -- Get subroutine name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 1;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   # -- Get file name
   my $filen = $_[0];

   # -- Check if file handle is defined
   unless ( -e $filen ){
      warn "${subname}__E> File not found: $filen\n";
      return;
   }

   # -- Check for gunzip
   unless ( `which gzip` ){
      warn "${subname}__E> gzip not found\n";
      return;
   }

   # -- Compress file
   if ( `gzip -f $filen` ){
      warn "${subname}__E> Failed compressing file: $filen\n";
      return;
   }

   # -- Check for compressed file
   my $gzfile = "${filen}.gz";
   unless ( -e $gzfile ){
      warn "${subname}__E> Could not find compressed file: $gzfile\n";
      return;
   }

   # -- Return gz filename
   return $gzfile;
}



sub GetArch {

   # -- Get subroutine name
   my $subname = GetSubrName();

   # -- Check arguments
   my $nargs = 0;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   # -- Get architecture
   my $arch = undef;
   chomp ($arch = `uname -m 2>&1`);

   # -- Check architecture
   unless ( $arch eq 'ia64' || $arch eq 'i686' ||
            $arch eq 'sun4u' || $arch eq 'x86_64' ){
      warn "${subname}__E> Do not understand architecture: $arch\n";
      return;
   }

   # -- Return
   return $arch;
}

sub MakeTMPDir {

   use IO::File;
   use File::Temp qw/ tempdir /;
   use File::Path;

   #--- Get subroutine name
   my $subname = GetSubrName();

   #--- Check arguments
   my $nargs = 0;

   die "${subname}__E> Insufficient arguments\n"
     unless (scalar(@_) == $nargs);

   my $tmpdir = tempdir();

   # --- return the tmpdir name
   return( $tmpdir );
}

sub GetFileSize {

   use File::stat;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__E> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my $fname = $_[0];

   my ( $fileparam, $filesize );

   undef $filesize;

   if ( -e $fname ){
      $fileparam = stat ( "$fname" );
      $filesize = $fileparam->size;
   }

   return $filesize;
}

sub CopyFile {

   use File::Basename;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 2;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__E> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my ($source, $target) = @_;

   # --- Check for source presence
   unless ( -e $source ){
      print "${subname}__E> Could not find source file $source\n";
      return;
   }

   # --- Make sure target directory exists
   my $targetdir = dirname( $target );
   unless ( -d $targetdir ){
      print "${subname}__E> Could not find target directory: $targetdir\n";
      return;
   }

   # --- Copy file
   my $retval = 0; my $count = 1;
   while ($retval or not compare_file_sizes($source, $target)) {
      $retval = system("cp -f $source $target 2>&1");
      sleep 5 if ( $count > 1 );
      $count++;
   }

   # --- Return value
   return $count;
}

sub compare_file_sizes {
   my ($source, $target) = @_;
   my $source_size = GetFileSize($source);
   my $target_size = GetFileSize($target);
   return (defined($source_size) and defined($target_size)
          and $source_size == $target_size);
}

sub OpenFile {

   use IO::File;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return ;
   }

   # --- Reassign input arguments
   my $filename = $_[0];

   # --- Check for presence
   unless ( -e $filename ){
      print "${subname}__W> Could not find file $filename\n";
      return;
   }

   # --- Open file and return filehandle
   my $fh;
   if (substr($filename, -3) eq '.gz' or substr($filename, -2) eq '.Z') {

      # If compressed or gzipped, open via gzip
      $fh = IO::File->new("gzip -dc $filename |");
   } elsif (substr($filename, -4) eq '.bz2') {

      # If bzipped, open via bzip2
      $fh = IO::File->new("bzip2 -dc $filename |");
   } else {
      $fh = IO::File->new("< $filename");
   }

   return $fh;
}

sub OpenNewFile {

   use IO::File;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return ;
   }

   # --- Reassign input arguments
   my $filename = $_[0];

   # --- Open file and return filehandle
   my $fh = IO::File->new("> $filename");

   return $fh;
}

sub TMPName {

   use File::Temp qw/ :mktemp  /;

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 0;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Find temporary filename that doesn't exist already
   my $name;
   do { $name = basename( mktemp("tmpfileXXXXX") ) }
      until ( ! -e $name );

   return $name;
}

# Create directory (and parent directories) if it does not exist.
# On failure, throws a fatal error.
sub MakeDirSys {
   my ($dir) = @_;

   eval { mkpath($dir, 0, 0755) };
   # For consistency with the rest of ModPipe, prepend E> to the error
   # message. Also add a stacktrace to make problems easier to track down.
   if ($@) {
     my $subname = GetSubrName();
     confess "${subname}__E> Cannot make directory $dir; $@";
   }
}

sub MakeDir {

   # --- Get subroutine name
   my $subname = GetSubrName();

   # --- Check arguments
   my $nargs = 1;

   unless ( scalar(@_) == $nargs ){
      print "${subname}__D> Insufficient arguments\n";
      return;
   }

   # --- Reassign input arguments
   my $dir = $_[0];

   # --- Create directory if it does not exist
   mkdir($dir, 0755) || return;

   # --- Return value
   return 1;
}


sub GetSubrName {
   use File::Basename;

   my $name = (caller(1))[3];
   unless ( $name ) { $name = basename($0) }
}

