#include "pdb_res.h"

using namespace std;

/*******************************************
RESIDUE PROPERTIES
********************************************/


// From atom structure to residue
void	prot_res::create_res( prot_atom *pa){
	int stack_first[NRES_MAX];
	int stack_last[NRES_MAX];

	nres = 0;	
	stack_first[0] = 0;
	for( int i = 1; i<pa->natm-1; i++){
		if( pa->atm[i].get_resnum() != pa->atm[i+1].get_resnum() ){
			stack_last[nres]	= i;
			stack_first[nres+1] = i+1;
			nres++;
		}
	}
	stack_last[nres] = pa->natm-1;
	nres++;

	res = new struct res[nres];

	for( int n = 0; n<nres; n++){
		int n0 = stack_last[n] - stack_first[n] + 1;
		res[n].natm = n0;
		res[n].atm = new class atom *[n0];
		for( int j = 0; j<n0; j++){
			res[n].atm[j] = &pa->atm[stack_first[n] + j]; 
		}
		strcpy(res[n].name, pa->atm[stack_first[n]].resname);
		res[n].num	 = pa->atm[stack_first[n]].resnum;
		res[n].chain = pa->atm[stack_first[n]].chain;
	}

	// now sequence
	cseq = new char[nres];

	for( int n = 0; n<nres; n++){
		cseq[n] = conv_3lett_1lett( res[n].name );
	}
}

/**********************************************
-- CNS --
**********************************************/
void prot_res::read_conservation(char file[])
{
	char buffer[1000];
	
	int n=0;
	int index;
	char residue;
	double cns;

	FILE *fp = fopen_read(file);

	while(fgets(buffer, 1000, fp)){
		if( strncmp(buffer, "# NSEQ", strlen("# NSEQ")-1)==0){ 
			char *ptr = strchr(buffer,'=');
			sscanf(ptr+1, "%d", &cns_nseq);
			//printf("cns_nseq %d %s\n", cns_nseq, ptr);
		}
		else if(buffer[0]=='#'){ 
			continue;
		}
		else{
			//printf("%s", buffer);
			sscanf(buffer, "%d %c %lf", &index, &residue, &cns);
			//printf("%3d %3d %c %8.3f\n", n, n1, c2, cns);
			if(n < nres) res[n].seq_cns = cns;
			n++;
		}
	}

	if(n != nres){
		fprintf(stderr, "\nprot_res::read_conservation	---> ERRORE Length in cns file: %d n: %d\n", nres, n);
		//exit(0);
	}

	fclose(fp);
}
/**********************************************
-- REAL_PATCH --
**********************************************/
void prot_res::read_real_patch(char file[])
{
	char buffer[10000];
	
	int n=0;

	if(nres==0){
		fprintf(stderr,"Error: No residues in memory; nres %d\n", nres);
	}
	real_patch = new int[nres];
	
	FILE *fp = fopen_read(file);

	fgets(buffer, sizeof(buffer), fp);
	
	for(unsigned int i=0; i<sizeof(buffer); i++){
		if(buffer[i] == '0' || buffer[i]=='2'){
			n++;
			real_patch[i] = (int)buffer[i]-'0';
		}
		else if(buffer[i]=='\n'){
			break;
		}
		else{
			fprintf(stderr,"Remove extra chars from file %s!!! %c\n", file, buffer[i]);
		}
	}

	if(n != nres){
		fprintf(stderr, "\nprot_res::read_real_patch	---> ERRORE Length in file %s: %d n: %d\n", file, nres, n);
		//exit(0);
	}

	fclose(fp);
}
/**********************************************
 -- STR CNS --
 **********************************************/
void prot_res::read_str_conservation(char file[])
{
	char buffer[1000];
	
	int n=0;
	int index;
	double cns;

	FILE *fp = fopen(file,"r");

	for(int i=0; i<nres; i++){			res[i].str_cns = 0.0;		}

	if(fp == NULL){		 return;	}

	while(fgets(buffer, 1000, fp)){
		if(buffer[0]=='#') continue;
		//printf("%s", buffer);
		sscanf(buffer, "%d %lf", &index, &cns);
		//printf("%3d %3d %c %8.3f\n", n, n1, c2, cns);
		while( res[n].num != index ){
			res[n].str_cns = 0.0;
			n++;
		}
		res[n].str_cns = cns;
		n++;
	}

	if(n > nres){
		fprintf(stderr, "\nprot_res::read_str_conservation ---> ERRORE Length in inf file: %d n: %d\n", nres, n);
		//exit(0);
	}

	fclose(fp);
}

/*---------------------------------------------
	--	ELEC POTENTIAL	--
 -----------------------------------------------*/

void prot_res::read_surf_ept(char *file)
{
	char buffer[1000];
	int n=0;
	atom atom;

	FILE *fp = fopen(file,"r");

	for(int i=0; i<nres; i++){			res[i].surf_ept = 0.0;		}

	if(fp == NULL){
		fprintf(stderr, "	File %s not read\n", file);
		return;	
	}

	while(fgets(buffer, 1000, fp)){
		if( strncmp(buffer, ATOM, 6)){ continue;}
		atom.read_temp(buffer);
		//printf( "	%d %f\n",n, atom.temp);
		res[n].surf_ept = atom.temp;
		n++;
	}

	if(n > nres){
		fprintf(stderr, "\nprot_res::read_str_conservation ---> ERRORE Length in inf file: %d n: %d\n", nres, n);
		//exit(0);
	}

	fclose(fp);
}


/**********************************************
INHERITE FROM ATOM
***********************************************/
void prot_res::determine_protrusion( prot_atom *pa)
{
	for(int n = 0; n < nres; n++){
		int natm_protr	= 0;
		int natm_cavity = 0;
		for(int i = 0; i < res[n].natm; i++){
			//if( atm[i].surf_flag	== 0) continue;
			if		 ( res[n].atm[i]->protr == 'P' ) natm_protr++;
			else if( res[n].atm[i]->protr == 'C' ) natm_cavity++;
			else if( res[n].atm[i]->protr == 'I' ) ;
			else{
				cerr << "Error! prot_res::determine_proterusion: protr not defined\n";
				cerr << n+1 << " " << i+1 << res[n].atm[i]->protr
					 << res[n].atm[i]->aname << " " <<res[n].atm[i]->anum << " "
					 << res[n].name << " " << res[n].num
					 << "\n";
				exit(0);
			}
		}
		if(natm_protr > natm_cavity) res[n].protr = 'P';
		else												 res[n].protr = 'C';
		//cout << natm_protr << " " <<natm_cavity<<"\n";
	}
}


/********************************************
	-----	 Interaction point		-----------
*********************************************/

// Calculate for every residue:
// - center of mass of buried and exposed residues
// - connection 
void prot_res::create_surfers( )
{
	if(nres == 0){
		cerr << "Error in prot_res::create_surfers - nres=0\n";
		exit(0);
	}

	for( int i=0; i<nres; i++){
		res[i].surf.ae = new int[res[i].natm];
		res[i].surf.ab = new int[res[i].natm];

		dvec_init(3, res[i].surf.vb, 0.0);
		dvec_init(3, res[i].surf.ve, 0.0);
		dvec_init(3, res[i].surf.norm, 0.0);

		res[i].surf.nb = 0;
		res[i].surf.ne = 0;

		for(int k=0; k< res[i].natm; k++){
			if(res[i].atm[k]->surf.surf_flag == 1){
				res[i].surf.ae[res[i].surf.ne] = k;
				res[i].surf.ne++;
				dvec_sum(3, 1.0, res[i].surf.ve, 1.0, res[i].atm[k]->xyz);
			}
			else{
				res[i].surf.ab[res[i].surf.nb] = k;
				res[i].surf.nb++;
				dvec_sum(3, 1.0, res[i].surf.vb, 1.0, res[i].atm[k]->xyz);
			}
		}
		if(res[i].surf.nb){ 
			dvec_scal(3, res[i].surf.vb, 1.0/res[i].surf.nb);
		}
		else{printf("Attention! Residue number %d no buried atoms!\n",i+1);}
		if(res[i].surf.ne){ 
			dvec_scal(3, res[i].surf.ve, 1.0/res[i].surf.ne);
			if(res[i].surf.nb){
				dvec_sum_xyz(3, res[i].surf.norm, 1.0, res[i].surf.ve, -1.0, res[i].surf.vb);
				dvec_norm(3, res[i].surf.norm);
			}
		}
	}
	flag_surf = 1;
}


/***************************************************
 *********	 Contact Map	 *************************
 ***************************************************/
/*
	dmap1 = distance between surfers, i.e. center of mass of exposed atoms
	dmap2 = distance between the closest exposed atoms
	dmap3 = effective distance is surfers + 2.A along the normal
*/


// Distance Map
void prot_res::create_dmap1()
{
	if(flag_surf == 0){
		cerr << "Error in prot_res::create_surfers - nres=0\n";
		exit(0);
	}

	dmap1 = alloc_dmat( nres, nres );

	dmat_init( nres, nres, dmap1, 999.0);

	double vec[3],x; 
 
	for	 (int i1=0; i1< nres ; i1++){
		if( res[i1].surf.ne > 0 ){
			for (int i2=i1+1; i2< nres	 ; i2++){
	if( res[i2].surf.ne > 0){
		dvec_sum_xyz(3, vec, 1.0, res[i1].surf.ve, -1.0, res[i2].surf.ve );
		x = dvec_length(3, vec);
		dmap1[i1][i2] = dmap1[i2][i1] = x;
		//printf("%d %d %f\n",i+1,j+1,x);
	}
			}
		}
	}
	flag_dmap1 = 1;
}
// Distance Map -- needs nearest neighbors atoms
void prot_res::create_dmap2()
{
	if(flag_surf == 0){
		cerr << "Error in prot_res::create_surfers - nres=0\n";
		exit(0);
	}

	dmap2 = alloc_dmat( nres, nres);

	dmat_init( nres, nres, dmap2, 999.0);

	int nmap=0;
	double vec[3],x,d0;	
	int k1,k2;
	int m1,m2;
	for	 ( int i1=0; i1< nres ; i1++){
		for ( int i2=i1+1; i2< nres	 ; i2++){
			if( res[i1].surf.ne != 0 && res[i2].surf.ne != 0 ){
				d0=999.0;
				// distance between two atoms is distance between the two closest atoms!!!
				for	( int j1=0; j1< res[i1].surf.ne ; j1++){
					for( int j2=0; j2< res[i2].surf.ne ; j2++){
						k1 = res[i1].surf.ae[j1];
						k2 = res[i2].surf.ae[j2];
						dvec_sum_xyz(3, vec, 1.0, res[i1].atm[k1]->xyz, -1.0, res[i2].atm[k2]->xyz );
						x = dvec_length(3, vec);
						if(d0>x){
							d0=x;
							m1 = k1;
							m2 = k2;
						}
					}
				}
				dmap2[i1][i2] = dmap2[i2][i1] = d0;
				nmap++;
				//printf("%d %d %f\n",i+1,j+1,x);
			}
		}
	}
	flag_dmap2 = 1;
}



/***************************************************
 *********	 NN LIST	 *************************
 ***************************************************/
/*
	Nearest Neighbor List - Only residues, whose distance is less than
	6.0A are nearest neighbor
*/

void prot_res::create_nst_dmap2(double dist)
{
	if(flag_dmap2 == 0){
		cerr << "Error in prot_res::create_surfers - nres=" <<nres <<"\n";
		exit(0);
	}

	if(res == NULL){
		res = new struct res[nres];
	}
	int tmp[nres];

	for(int i=0; i<nres;i++){
		int n=0;
		for(int j=0; j<nres; j++){
			if(dmap2[i][j] < dist){
				tmp[n] = j;
				n++;
			}
		}
		res[i].nst.n = n;
		res[i].nst.v = new int[n];
		for(int j=0; j<n; j++){
			res[i].nst.v[j] = tmp[j];
		}
	}

	flag_nst = 1;
}

/***************************************************
 *********	 NN MAP	 *************************
 ***************************************************/

void prot_res::create_imap()
{
	imap = alloc_imat(nres, nres);

	for(int i=0; i<nres;i++){
		if(res[i].nst.n != 0){
			//printf(">%4d \n", i+1);
			breadth_first_search(i, imap[i]);
		}
	}
}

void prot_res::breadth_first_search(int s, int *dist)
{
	int flag[nres];
	ivec_init(nres, flag, 0);
	ivec_init(nres, dist, 0);

	class queue q;
	q.init(nres);
	q.put(s);
	flag[s] = 1;

	int u,v;
	while(q.size() != 0){
		u = q.get();
		for(int j=0; j< res[u].nst.n; j++){
			v = res[u].nst.v[j];
			if( flag[v] == 0 ){
				dist[v] = dist[u]+1;
				q.put(v);
				flag[v] = 1;
			}
		}
		flag[u]=2;
	}
}
 

/*************************************************
Divergence is defined dv1/dx1 + dv2/dx2 + dv3/dx3
where v is the normal to the surface
**************************************************/
void prot_res::evaluate_divergence()
{
	double divergence(double v1[], double v2[], double x1[], double x2[]);
	
	int i2;
	for(int i=0; i<nres;i++){
		int ndiv=0;
		if(res[i].surf.ne*res[i].surf.nb == 0){
			res[i].surf.div = 0.0;
			continue;
		}
		for(int j=0; j< res[i].nst.n; j++){
			i2 = res[i].nst.v[j];
			//printf("%4d %4d	", i, i2);
			res[i].surf.div += divergence(res[i].surf.norm, res[i2].surf.norm, res[i].surf.ve, res[i2].surf.ve);
			ndiv++;
		}
		res[i].surf.div /= (double)ndiv;
	}
}

double divergence(double v1[], double v2[], double x1[], double x2[])
{
	double y1[3],y2[3];

	double dist1 = dvec_distance(3, x1, x2);
	dvec_sum_xyz(3, y1, 1.0, v1, 1.0, x1);
	dvec_sum_xyz(3, y2, 1.0, v2, 1.0, x2);
	double dist2 = dvec_distance(3, y1, y2);

	//printf("distance %f %f\n", dist1, dist2);
	return (dist2/dist1 - 1.0)*10.0;
}


/**************************************************

***************************************************/
void prot_res::calculate_charge_exposed()
{
	for	 ( int i1=0; i1< nres ; i1++){
		for	( int j1=0; j1< res[i1].surf.ne ; j1++){
			int k1 = res[i1].surf.ae[j1];
			double charge = res[i1].atm[k1]->charge;
			res[i1].charge_exp += charge;
		}
	}
}

/**************************************************

***************************************************/
void prot_res::calculate_temp_exposed()
{
	for	 ( int i1=0; i1< nres ; i1++){
		for	( int j1=0; j1< res[i1].surf.ne ; j1++){
			int k1 = res[i1].surf.ae[j1];
			double temp = res[i1].atm[k1]->temp;
			res[i1].temp_exp += temp;
		}
	}
}
/**************************************************

***************************************************/
void prot_res::calculate_psa()
{
	for	 ( int i1=0; i1< nres ; i1++){
		for	( int j1=0; j1< res[i1].natm ; j1++){
			double sol = res[i1].atm[j1]->sol;
			res[i1].psa += sol;
		}
	}
}


/**************************************************
 --- WRITE SOMETHING --
***************************************************/

//
void prot_res::write_res(char file[])
{
	FILE *fp = fopen_write(file);
	fprintf(fp, "N. of res: %d\n", nres);
	fflush(fp);
	for( int i=0; i<nres; i++){
		fprintf(fp, "# %4d %4d %3d %s", i+1, res[i].num, res[i].natm, res[i].name); 
		fflush(fp);
		for(int k=0; k< res[i].natm; k++){
			fprintf(fp, "%d:%s,", k+1,res[i].atm[k]->aname);
		}
		fprintf(fp,"\n");
	}	
	fclose(fp);
}

// Print basic information

void prot_res::write_surfers(char file[])
{
	FILE *fp = fopen_write(file);

	for(int i=0; i<nres; i++){ 
		fprintf(fp, "# %3d	%4d %s	 (%5.1f %5.1f %5.1f)	 %2d || ",
			i+1, res[i].num, res[i].name, res[i].surf.ve[0], res[i].surf.ve[1], res[i].surf.ve[2], res[i].surf.ne);
		for(int j=0; j<res[i].surf.ne; j++){
			fprintf(fp, "%2d ", res[i].surf.ae[j]);
		}
		fprintf(fp,"\n");
		if(res[i].surf.nb){
			fprintf(fp, "# %3d	%4d %s	 (%5.1f %5.1f %5.1f)	 %2d || ",
				i+1, res[i].num, res[i].name, res[i].surf.vb[0], res[i].surf.vb[1], res[i].surf.vb[2], res[i].surf.nb);
			for(int j=0; j<res[i].surf.nb; j++){ 
				fprintf(fp, "%2d ", res[i].surf.ab[j]);
			}
			fprintf(fp,"\n");
		}
	}
	fclose(fp);
}

//Print Map
void prot_res::write_dmap(char *file, int num)
{
	double **map;

	if(num==1){
		map = dmap1;
	}
	else if(num==2){
		map = dmap2;
	}
	else{
		fprintf(stderr,"	ERROR in prot_res::write_dmap --> num has to be 1 or 2\n");
	}

	FILE *fp = fopen_write(file);

	fprintf(fp,"%d	 %d\n",nres, nres);

	fprintf(fp,"#");
	for(int j=0; j<nres; j++){
		fprintf(fp,"	%6d", j+1);
	}
	fprintf(fp,"\n");
	
	for(int i=0; i<nres; i++){
		fprintf(fp,"# %3d ", i+1);
		for( int j=0; j<nres; j++){
			fprintf(fp," %6.2f", map[i][j]);
		}
		fprintf(fp,"\n");
	}
	fclose(fp);
}


void prot_res::write_rprotr(char file[])
{
	FILE *fp = fopen_write(file);

	fprintf(fp,"%d\n",nres);
	for(int i=0; i<nres; i++){ 
		fprintf(fp, "%5d %c", i+1, res[i].protr);
		fprintf(fp,"\n");
	}
	fclose(fp);
}

void prot_res::read_rprotr(char file[])
{
	FILE *fp = fopen_read(file);

	int nres1;
	fscanf(fp,"%d",&nres1);

	if(nres1!=nres){
		fprintf(stderr," ERRORR in prot_res::raed_rprotr nres1 %d = nres %d \n", nres1,nres);
	}

	for(int i=0; i<nres; i++){ 
		fscanf(fp, "%5d %c", &i, &(res[i].protr));
	}
	fclose(fp);
}



//Print Map
void prot_res::read_dmap2(char *file, int debug)
{
	char buffer[50000];
	int nres1,nres2;

	FILE *fp = fopen_read(file);

	fscanf(fp, "%d%d", &nres1, &nres2);

	if(nres1==nres2){ 
		nres=nres1;
	}
	else{
		fprintf(stderr,"	 ERORR in prot_res::read_dmap\n");
	}
	if(debug>0)cout << nres << " " << nres1 << " " <<nres2 << flush;

	dmap2 = alloc_dmat( nres, nres);

	char *buffer_ptr = fgets(buffer, 50000, fp);
	buffer_ptr = fgets(buffer, 50000, fp);

	if(buffer_ptr==NULL){fprintf(stderr,"	 ERORR in prot_res::read_dmap variable not read\n");}

	if( debug > 0 ){
		cout << buffer << "\n";
	}	

	for(int i=0; i<nres; i++){
		int num;
		char ch;
		fscanf(fp, "%c %3d ", &ch, &num);
		//printf("%4d ",i);
		for( int j=0; j<nres; j++){
			fscanf(fp," %lf ", &dmap2[i][j]);
			//printf("	%d %f",j, dmap2[i][j]);
		}
		//printf("\n");
	}
	
	buffer_ptr = fgets(buffer, 50000, fp);
	if(buffer_ptr!=NULL){
		fprintf(stderr," Error in prot::read_dmap2 <%s>\n",buffer);
	}

	fclose(fp);

	flag_dmap2 = 1;
}

void prot_res::write_nn_list(char *file)
{
	FILE *fp = fopen_write(file);

	fprintf(fp," %d\n",nres);

	for(int i=0; i<nres; i++){
		fprintf(fp,"%3d	%3d	", i+1, res[i].nst.n);
		for( int j=0; j<res[i].nst.n; j++){
			fprintf(fp,"%3d ", res[i].nst.v[j]+1);
		}
		fprintf(fp,"\n");
	}
	fclose(fp);
}

//rasmol
void prot_res::rasmol_exposed_residues()
{
	printf("select ");
	for( int i=0; i<nres; i++){
		if(res[i].surf.ne > 0){
			printf("%d,", res[i].num );
		}
	}	
	printf("\n\n");
}



void prot_res::rasmol_nn_list(char *file, char str[])
{
	FILE *fp = fopen_write(file);

	fprintf(fp, "%s", str);
	for(int i=0; i<nres; i++){
		if(res[i].nst.n == 0) continue;
		fprintf(fp, "select all\n");
		fprintf(fp, "color grey\n");
		fprintf(fp, "select %d\n", res[i].num);
		fprintf(fp, "color red\n");
		for(int j=0; j< res[i].nst.n ; j++){
			int k = res[i].nst.v[j];
			fprintf(fp, "select %d\n", res[k].num);
			fprintf(fp, "color orange\n");
		}
		fprintf(fp, "pause\n");
	}
}
//Print Map
void prot_res::write_imap(char *file)
{
	FILE *fp = fopen_write(file);

	fprintf(fp,"%d	 %d\n",nres, nres);


	for(int i=0; i<nres; i++){
		for( int j=0; j<nres; j++){
			fprintf(fp," %d", imap[i][j]);
		}
		fprintf(fp,"\n");
	}
	fclose(fp);
}

/************************************
	 WRITE PDB FUNTIONS based on atom
*************************************/
void prot_res::write_pdb_divergence(char *file, prot_atom *pa)
{
	for(int i=0; i<nres; i++){
		for(int j=0; j<res[i].natm; j++){
			res[i].atm[j]->temp = res[i].surf.div;
		}
	}
	pa->write_atom(file);
}

void prot_res::write_pdb_protruding(char *file, prot_atom *pa)
{
	for(int i=0; i<nres; i++){
		//cout << i+1 <<res[i].protr <<"\n";
		for(int j=0; j<res[i].natm; j++){
			if(res[i].protr == 'P')			res[i].atm[j]->temp = 1.0;
			else if(res[i].protr == 'C') res[i].atm[j]->temp = 2.0;
		}
	}
	pa->write_atom(file);
}

void prot_res::write_pdb_random_residue(char *file, prot_atom *pa)
{
	for(int i=0; i<nres; i++){
		double r = ran3(&IDUM);
		for(int j=0; j<res[i].natm; j++){
			res[i].atm[j]->temp = r;
		}
	}
	pa->write_atom(file);
}

void prot_res::write_pdb_resnorm(char *file, prot_atom *pa)
{
	for(int i=0; i<nres; i++){
		for(int j=0; j<res[i].natm; j++){
			dvec_cp(3, res[i].atm[j]->surf.norm, res[i].surf.norm);
		}
	}
	pa->write_pdb_norm(file);
}

