/*******************************************************************
 *	BINDING SITE PREDICTION PROGRAM
 *
 *	Input: features (.cns,...)
 *	Output: files con binding sites (1,0)
 *
 *	Descritpion: this program
 *
 *
 *
 *
 ********************************************************************/
#include "bsp.h"


int    SIZE_AVE    = 14;
int    SIZE_STD    = 3;
int    LOG         = 1;
int    N_STEP      = 10;
int    NMC_MAX     = 1;
double TEMP        = 1.0;    // Simulated Annealling: Temperature
double TEMP_FCTR   = 0.999; // Simulated Annealing: Temperature
double AACCESS     = 2.0;     // Accessibility in Angstrom square
double NN_DIST     = 6.0;     // Cutoff for nearest neighbour

int SCORE_FLAG;
double WEIGHT[ENERGY_MAX];
double WEIGHT_AVE[ENERGY_MAX];
double WEIGHT_STD[ENERGY_MAX];

int CORE  = 0;
int SURF  = 1;
int PATCH = 2;

int LENGTH;

int nmc_counter;
int DEBUG = 0;
int DEBUG_PDB = 0;

//OUTPUT
char FILE_OUT1[1000];
char FILE_OUT2[1000];
char FILE_OUT1A[1000];
char FILE_OUT2A[1000];
char FILE_OUT1B[1000];
char FILE_OUT2B[1000];
char FILE_OUT1C[1000];
char FILE_OUT2C[1000];

FILE *fp_mc1;
FILE *fp_mc2;
FILE *fp_mc3;
FILE *fp_mc4;

char FILE_MC1[1000];
char FILE_MC2[1000];
char FILE_MC3[1000];
char FILE_MC4[1000];
char FILE_ZSCORE[1000];
char FILE_INPUT[1000];
char FILE_LIG[1000];

// Options
char Sol_file[1000];		 
char Charge_file[1000];		 
char Prt_file[1000];		 
char Cns_file[1000];
char Inf_file[1000];
char Ept_file[1000];
char Psa_file[1000];
char Lig_file[1000];
char Dmap2_file[1000];
char Rprotr_file[1000];
char OUT_DIR[1000];

char Usage[1000];

prot_atom PA;
prot_res PR;
site Lig;
site *Site;

// ------ RADNOM PATCHES OR Z SCORE

site RPATCH[N_RANDOM_PATCHES];
double GG_PROTEIN_AVE[ENERGY_MAX];
double GG_PROTEIN_STD[ENERGY_MAX];

/*********************************************************************
 *****************		 MAIN		***************************************
 ************************************************************/
int main(int argc, char *argv[])
{
	void read_input(char *s);
	void eval_zscore(site *);
	void write_param(void);
	void prepare(void);
	void optimize(void);
	void ligand(void);

	//ios::sync_with_stdio();
	read_input(argv[1]);
	write_param();
	prepare();
	ligand();
	eval_zscore(RPATCH);
	optimize();
}


/*-------------------------------------------------------------
	--								OPTIONS ROUTINE
	--------------------------------------------------------------*/


void read_input(char *file)
{	
	void init_weights(char *string, double v[]);
	// Initialization before parameter are assigned
	//weight are zero-initialized
	for(int i=0; i<ENERGY_MAX; i++){
		WEIGHT[i] = 0.0;
	}
	//OUT_DIR is ./ initizialized
	strcpy(OUT_DIR,".");
	


	FILE *fp = fopen(file, "r");
	if(fp==NULL){
		fprintf(stderr, "Could not open pippo!\n");
		exit(8);
	}
	char buffer[10000];
	char key[10000];
	char var[10000];

	while( fgets(buffer, sizeof(buffer), fp) ){
		if(buffer[0] == '#'){
			continue;
		}
		sscanf(buffer, "%s %s", key, var);
		
		if(strcmp(key,"sol_file")==0){
			strcpy(Sol_file, var);
		}
		else if(strcmp(key,"prt_file")==0){
			strcpy(Prt_file, var);
		}
		else if(strcmp(key,"crg_file")==0){
			strcpy(Charge_file, var);
		}
		else if(strcmp(key,"cns_file")==0){
			strcpy(Cns_file, var);
		}
		else if(strcmp(key,"ept_file")==0){
			strcpy(Ept_file, var);
		}
		else if(strcmp(key,"psa_file")==0){
			strcpy(Psa_file, var);
		}
		else if(strcmp(key,"lig_file")==0){
			strcpy(Lig_file, var);
		}
		else if(strcmp(key,"rprotr_file")==0){
			strcpy(Rprotr_file, var);
		}
		else if(strcmp(key,"out_dir")==0){
			strcpy(OUT_DIR, var);
		}
		else if(strcmp(key,"temp")==0){
			sscanf(var, "%lf", &TEMP);
		}
		else if(strcmp(key,"temp_fctr")==0){
			sscanf(var, "%lf", &TEMP_FCTR);
		}
		else if(strcmp(key,"nmc")==0){
			sscanf(var, "%d", &NMC_MAX);
		}
		else if(strcmp(key,"nstep")==0){
			sscanf(var, "%d", &N_STEP);
		}
		else if(strcmp(key,"size_ave")==0){
			sscanf(var, "%d", &SIZE_AVE);
		}
		else if(strcmp(key,"size_std")==0){
			sscanf(var, "%d", &SIZE_STD);
		}
		else if(strcmp(key,"debug")==0){
			sscanf(var, "%d", &DEBUG);
		}
		else if(strcmp(key,"debug_pdb")==0){
			sscanf(var, "%d", &DEBUG_PDB);
		}
		else if(strcmp(key,"score_flag")==0){
			sscanf(var, "%d", &SCORE_FLAG);
		}
		// format: 1.1:2.23:1.24:1.0:0.0:0.0:1.23
		else if(strcmp(key,"weights")==0){
			init_weights(var,WEIGHT);
		}
		else if(strcmp(key,"weights_ave")==0){
			init_weights(var,WEIGHT_AVE);
		}
		else if(strcmp(key,"weights_std")==0){
			init_weights(var,WEIGHT_STD);
		}
		else{
			fprintf(stderr, "variable not recognized!\n%s\n",buffer);
			exit(8);
		}
	}
	fclose(fp);

	if( (access(OUT_DIR, F_OK)) != 0 ){
		fprintf(stderr,"\nNo %s! Please create directory!\n\n", OUT_DIR);
		exit(8);
	}
	// Initialization after parameters are assigned
 	strcpy(FILE_OUT1, OUT_DIR);
	strcat(FILE_OUT1, "/patch1.dat");
 	strcpy(FILE_OUT2, OUT_DIR);
	strcat(FILE_OUT2, "/patch2.dat");

 	strcpy(FILE_OUT1A, OUT_DIR);
	strcat(FILE_OUT1A, "/patch1a.dat");
 	strcpy(FILE_OUT2A, OUT_DIR);
	strcat(FILE_OUT2A, "/patch2a.dat");
 	strcpy(FILE_OUT1B, OUT_DIR);
	strcat(FILE_OUT1B, "/patch1b.dat");
 	strcpy(FILE_OUT2B, OUT_DIR);
	strcat(FILE_OUT2B, "/patch2b.dat");
 	strcpy(FILE_OUT1C, OUT_DIR);
	strcat(FILE_OUT1C, "/patch1c.dat");
 	strcpy(FILE_OUT2C, OUT_DIR);
	strcat(FILE_OUT2C, "/patch2c.dat");
	
 	strcpy(FILE_MC1, OUT_DIR);
	strcat(FILE_MC1, "/mc1.dbg");
 	strcpy(FILE_MC2, OUT_DIR);
	strcat(FILE_MC2, "/mc2.dbg");
 	strcpy(FILE_MC3, OUT_DIR);
	strcat(FILE_MC3, "/mc3.dbg");
 	strcpy(FILE_MC4, OUT_DIR);
	strcat(FILE_MC4, "/mc4.dbg");
	
 	strcpy(FILE_ZSCORE, OUT_DIR);
	strcat(FILE_ZSCORE, "/zscore.dat");
	
	strcpy(FILE_INPUT, OUT_DIR);
	strcat(FILE_INPUT, "/input.dat");
	
	strcpy(FILE_LIG, OUT_DIR);
	strcat(FILE_LIG, "/ligand.dat");
}

void init_weights(char *string, double *v)
{
	char *first_ptr = string;
	char *last_ptr;
	
	int i;
	
	for(i=0;  ; i++){
		last_ptr = strchr(first_ptr,':');
			
		if(last_ptr == NULL){
			last_ptr = strchr(first_ptr,'\0');
			sscanf(first_ptr, "%lf", &v[i]);
			break;
		}
		else{
			*last_ptr = '\0';
			sscanf(first_ptr, "%lf", &v[i]);
			first_ptr = last_ptr+1;
		}
		//fprintf(stdout, "FIRST %x\n", (unsigned int)first_ptr);
		//fprintf(stdout, "LAST  %x\n", (unsigned int)last_ptr);
		//fprintf(stdout, "STRING  %s\n", (unsigned int)first_ptr);
		//fprintf(stdout, "STRING  %s\n", (unsigned int)string);
	}

	if(i!=ENERGY_MAX - 1){
		fprintf(stderr,"ERROR in weights: i=%d ENERGY_MAX:%d\n", i, ENERGY_MAX);
		exit(1);
	}
	
	return;
}

void write_param()
{
	FILE *fp = fopen_write(FILE_INPUT);
	
	fprintf(fp, "Here is the input of bsp\n\n");
	// Files
	fprintf(fp, "SOL_FILE:\t%s\n",	Sol_file);
	fprintf(fp, "SOL_FILE:\t%s\n",	Charge_file);
	fprintf(fp, "SOL_FILE:\t%s\n",	Prt_file);
	fprintf(fp, "CNS_FILE:\t%s\n",	Cns_file);
	fprintf(fp, "EPT_FILE:\t%s\n",	Ept_file);
	fprintf(fp, "PSA_FILE:\t%s\n",	Psa_file);
	fprintf(fp, "LIG_FILE:\t%s\n",	Lig_file);
	// Parameters
	fprintf(fp, "SIZE_AVE:\t%d\n",	SIZE_AVE);
	fprintf(fp, "NMC_MAX:\t%d\n",	NMC_MAX);
	fprintf(fp, "N_STEP:\t%d\n",	N_STEP);
	fprintf(fp, "TEMP:\t%f\n",	    TEMP);
	fprintf(fp, "TEMP_FCTR:\t%f\n",	TEMP_FCTR);
	fprintf(fp, "AACCESS:\t%f\n",	AACCESS);
	// Weights
	for(int i=0; i<ENERGY_MAX; i++){
		fprintf(fp, "WEIGHT(%d):\t%f\n",i,WEIGHT[i]);
	}
	fclose(fp);
}

/*************************
Prepare the system PA and PR
**************************/

void prepare(void){

	int result;
	
	if( (result = access(Sol_file, F_OK)) == 0 ){
		// store file in buffer
		PA.natm = read_buffer( Sol_file, BUFFER );
		// from buffer read atom
		PA.create_atom_sol( BUFFER);
	
		// assign a flag if atom is in surface or not
		PA.atom_surface(AACCESS);
		// create nerest neighbors list
		PA.create_nst(NN_DIST);
		if(DEBUG_PDB){ PA.write_count("pdb_atom_count.dat" );}
		// determine atomic protrusion
		PA.atom_determine_protrusion();
		if(DEBUG_PDB){ PA.write_pdb_protrusion("pdb_atom_protr.pdb" );} 
		if(DEBUG_PDB){ PA.write_atom("pdb_atom.dat" );}

		// create residues
		PR.create_res(&PA);
		PR.calculate_psa();
		// assign buried atoms, exposed atoms and normal
		PR.create_surfers();
		// create distance map from the two closest atoms
		PR.create_dmap2();
		// create nearest neighbors list
		PR.create_nst_dmap2(6.0);
		// determine protrusion depending how many atoms are in protrusion
		PR.determine_protrusion(&PA);
		PR.evaluate_divergence();
		// write output
		PR.write_rprotr("pdb_rprotr.dat");   
		PR.write_pdb_protruding("pdb_res_protr.pdb", &PA);
	
		PR.write_nn_list("pdb_nn_list.dat");
	}
	else{ fprintf(stderr, "\nError: sol_file has to be specified!!!\n\n"); exit(0);}

	// store file in buffer
	int n;
	if( (n = read_buffer( Prt_file, BUFFER )) != PA.natm ){
		fprintf(stderr, "ERROR natm %d != n %d\n", PA.natm, n);
	}
	else{
  		for( int i=0; i<PA.natm; i++){
    		PA.atm[i].read_temp(BUFFER[i]);
  		}
		PR.calculate_temp_exposed();
	}
	// store file in buffer
	if( (n = read_buffer( Charge_file, BUFFER )) != PA.natm ){
  			fprintf(stderr, "ERROR natm %d != n %d\n", PA.natm, n);
	}
	else{
  		for( int i=0; i<PA.natm; i++){
    		PA.atm[i].read_charge(BUFFER[i]);
  		}
		PR.calculate_charge_exposed();
	}

	if( (result = access(Cns_file, F_OK)) == 0 ){
		PR.read_conservation( Cns_file);
	}
	if( (result = access(Inf_file, F_OK)) == 0 ){
		PR.read_str_conservation( Inf_file);
	}
	if(DEBUG>0){printf("Reading Ept_file\n");}
	
	if( (result = access(Ept_file, F_OK)) == 0 ){
		PR.read_surf_ept( Ept_file);
	}	
	LENGTH = PR.number_of_res();
	if(DEBUG>0) printf("LENGTH %d \n",LENGTH);

// Lig.create_core( &PR );
	if ((result = access(Lig_file, F_OK)) == 0 ){
		//Lig.read_standard( Lig_file );
		PR.read_real_patch( Lig_file );
	}
}

/*******************
----- ZSCORE ----
**********************/

void eval_zscore(site *site){
	
	double vtmp[ENERGY_MAX][N_RANDOM_PATCHES];
	
	for(int i=0; i<N_RANDOM_PATCHES; i++){
 		int num_of_res = Lig.patch.n;
		if(num_of_res == 0){
			num_of_res = SIZE_AVE;
		}
		site[i].create_patch( &PR, num_of_res );
		site[i].score_all(&PR);
		for(int j=0; j<ENERGY_MAX; j++){
			vtmp[j][i] = site[i].GG[j];
		}
	}
	for(int j=0; j<ENERGY_MAX; j++){
		GG_PROTEIN_AVE[j] = dvec_stat(N_RANDOM_PATCHES, vtmp[j], &GG_PROTEIN_STD[j]);
	}

		
	//if(DEBUG>0){
	FILE *fp = fopen_write(FILE_ZSCORE);
	for(int j=0; j<ENERGY_MAX; j++){
		fprintf(fp, "# %d\t%f\t%f\n", j, GG_PROTEIN_AVE[j], GG_PROTEIN_STD[j] );
	}
	for(int i=0; i<N_RANDOM_PATCHES; i++){
		for(int j=0; j<ENERGY_MAX; j++){
			fprintf(fp,"%f\t", vtmp[j][i]);
		}
		fprintf(fp,"\n");
	}
	fclose(fp);
	//}
}

/**************************************
optimize
**************************************/

void optimize(void){
	if(DEBUG>0){
		fp_mc1 = fopen_write( FILE_MC1 );
		fp_mc2 = fopen_write( FILE_MC2 );
		fp_mc3 = fopen_write( FILE_MC3 );
		fp_mc4 = fopen_write( FILE_MC4 );
	}
		
	Site = new site[NMC_MAX];

	FILE *fp_patch1 = fopen_write( FILE_OUT1 );
	FILE *fp_patch2 = fopen_write( FILE_OUT2 );
	FILE *fp_patch1a = fopen_write( FILE_OUT1A );
	FILE *fp_patch2a = fopen_write( FILE_OUT2A );
	FILE *fp_patch1b = fopen_write( FILE_OUT1B );
	FILE *fp_patch2b = fopen_write( FILE_OUT2B );
	FILE *fp_patch1c = fopen_write( FILE_OUT1C );
	FILE *fp_patch2c = fopen_write( FILE_OUT2C );

	for(nmc_counter=0; nmc_counter < NMC_MAX; nmc_counter++){
  		int num_of_res = (int)uniform_distribution((double)SIZE_AVE, (double)SIZE_STD);
		Site[nmc_counter].create_patch( &PR, num_of_res );
		Site[nmc_counter].score( &PR );
	
		fprintf(fp_patch1, "%d\t%f\t%f\t",nmc_counter,Site[nmc_counter].G, Site[nmc_counter].GG[ENERGY_OVERLAP]);
		Site[nmc_counter].fprintf_flag(fp_patch1);
		fflush(fp_patch1);
		
		fprintf(fp_patch1a, "%d\t%f\t",nmc_counter,Site[nmc_counter].G);
		Site[nmc_counter].fprintf_GG(fp_patch1a);
		fflush(fp_patch1a);
		
		fprintf(fp_patch1b, "%d\t%f\t",nmc_counter,Site[nmc_counter].G);
		Site[nmc_counter].fprintf_GZ(fp_patch1b);
		fflush(fp_patch1b);
		
		fprintf(fp_patch1c, "%d\t%f\t",nmc_counter,Site[nmc_counter].G);
		Site[nmc_counter].fprintf_GW(fp_patch1c);
		fflush(fp_patch1c);

		// -- Begin Optimization --
		
		Site[nmc_counter].optimize( &PR);
		
		// -- End   Optimization --
		
		fprintf(fp_patch2, "%d\t%f\t%f\t",nmc_counter,Site[nmc_counter].G, Site[nmc_counter].GG[ENERGY_OVERLAP]);
		Site[nmc_counter].fprintf_flag(fp_patch2);
		fflush(fp_patch2);
		
		fprintf(fp_patch2a, "%d\t%f\t",nmc_counter,Site[nmc_counter].G);
		Site[nmc_counter].fprintf_GG(fp_patch2a);
		fflush(fp_patch2a);
		
		fprintf(fp_patch2b, "%d\t%f\t",nmc_counter,Site[nmc_counter].G);
		Site[nmc_counter].fprintf_GZ(fp_patch2b);
		fflush(fp_patch2b);
		
		fprintf(fp_patch2c, "%d\t%f\t",nmc_counter,Site[nmc_counter].G);
		Site[nmc_counter].fprintf_GW(fp_patch2c);
		fflush(fp_patch2c);
	}
	
	fclose(fp_patch1);
	fclose(fp_patch2);
	fclose(fp_patch1a);
	fclose(fp_patch2a);
	fclose(fp_patch1b);
	fclose(fp_patch2b);
	fclose(fp_patch1c);
	fclose(fp_patch2c);
	
	if(DEBUG>0){
		fclose( fp_mc1 );
		fclose( fp_mc2 );
		fclose( fp_mc3 );
		fclose( fp_mc4 );
	}
}

/**************************************************
	REAL LIGAND
***************************************************/

void ligand(void){
	
	int result;
	
	if ((result = access(Lig_file, F_OK)) == 0 ){
		Lig.inherite_ligand(&PR);
		Lig.score(&PR);
		FILE *fp = fopen_write(FILE_LIG);
		Lig.fprintf_all(fp);
		fclose(fp);
	}
}

/*---------------------------------------------------------
 -----------------------------------------------------------*/

void site::fprintf_all(FILE *fp)
{
	fprintf_flag(fp);
	fprintf_nres(fp);
	fprintf(fp, "TOTAL_ENERGY\t%f\n", G);

	fprintf(fp, "VECTOR_GG\t");
	for(int j=0; j<ENERGY_MAX; j++){
		fprintf(fp, "%f\t", GG[j]);
	}
	fprintf(fp, "\n");
	
	fprintf(fp, "VECTOR_GZ\t");
	for(int j=0; j<ENERGY_MAX; j++){
		fprintf(fp, "%f\t", GZ[j]);
	}
	fprintf(fp, "\n");
	
	fprintf(fp, "VECTOR_GW\t");
	for(int j=0; j<ENERGY_MAX; j++){
		fprintf(fp, "%f\t", GW[j]);
	}
	fprintf(fp, "\n");
}

/*---------------------------------------------------------
	Auxiliary print routines
 -----------------------------------------------------------*/

void site::fprintf_statistics(FILE *fp)
{
	fprintf(fp, "%6d ", overlap_lig);
	fprintf(fp, "%6d ", patch.n);
	fprintf(fp, "%6d ", nres_protr);
	fprintf(fp, "%6d ", nres_cavity);
	fprintf(fp, "%6.3f ", cns_patch);
	fprintf(fp, "%6.3f ", inf_patch);
	fprintf(fp, "%6d ", num_tot);
	fprintf(fp, "%6d ", num_buried);
	fprintf(fp, "%6d ", num_exposed);
	fprintf(fp, "%6d ", num_O);
	fprintf(fp, "%6d ", num_C);
	fprintf(fp, "%6d ", num_N);
	fprintf(fp, "%6.3f ", hydro);
	fprintf(fp, "\n");
}


// Print flag
void site::fprintf_flag(FILE *fp)
{
	for(int i=0; i<LENGTH; i++){
		fprintf(fp,"%1d", flag_patch[i]);
	}
	fprintf(fp, "\n");
}

// Print nres
void site::fprintf_nres(FILE *fp)
{
	fprintf(fp,"%d\t", core.n);
	fprintf(fp,"%d\t", surf.n);
	fprintf(fp,"%d\n", patch.n);
}

// Print GG
void site::fprintf_GG(FILE *fp)
{
	for(int i=0; i<ENERGY_MAX; i++){
		fprintf(fp,"%f\t", GG[i]);
	}
	fprintf(fp, "\n");
}

// Print GZ
void site::fprintf_GZ(FILE *fp)
{
	for(int i=0; i<ENERGY_MAX; i++){
		fprintf(fp,"%f\t", GZ[i]);
	}
	fprintf(fp, "\n");
}

// Print GW
void site::fprintf_GW(FILE *fp)
{
	for(int i=0; i<ENERGY_MAX; i++){
		fprintf(fp,"%f\t", GW[i]);
	}
	fprintf(fp, "\n");
}

